# $Id: diffusion.R,v 1.2 2019/11/03 16:37:59 stefan Exp stefan $
# Estimation of diffusion coefficients
# Poling, B. E.; Prausnitz, J. M. and O’Connell, J. P.
# The Properties of Gases and Liquids, McGraw-Hill, 2004


# Chapman Enskok model

# Constants
k <- 1.38064852E-23
p0 <- 1.01325
T0 <- 273.15

# Neufeld et al. (1972)
A <- 1.06036
B <- 0.15610
C <- 0.19300
D <- 0.47635
E <- 1.03587
F <- 1.52996
G <- 1.76474
H <- 3.89411

# Collision integral
omega <- function (Theta) {
	if (Theta < 0.3 || Theta > 100) {
		cat ("omega: Theta out of range: ")
		cat(Theta)
		cat("\n")
		stop()	
	}
	retrun <- A/Theta^B + C/exp(D*Theta) + E/exp(F*Theta) + G/exp(H*Theta)
}

# Diffusion coefficient Chapman Enskok model
Dce <- function(mA, mB, eA, eB, sA, sB, T=T0, p=p0)  {
	mAB <- 2 / (1/mA + 1/mB)
	eAB <- sqrt(eA * eB)
	sAB <- (sA + sB)/2
	theta <- T / eAB
	omega0 <- omega(theta)
	return <- (0.00266*T^(1.5)/(p*sqrt(mAB)*sAB^2*omega0))
}


dodt <- function(Theta) {
	return <- - (A*B/Theta^(B+1) + C*D/exp(D*Theta) + E*F/exp(F*Theta) + G*H/exp(H*Theta))
}

# Temperature coefficient
b <- function(eA, eB, T=T0) {
	eAB <- sqrt(eA * eB)
	Theta <- T / eAB
	return <- 1.5 - Theta/omega(Theta)*dodt(Theta)
}

# Example 11-1: Estimate the diffusion coefficient for the system N2-CO2 at 590 K and 1 bar.
# Result: 0.52 cm²/s
stopifnot(round(Dce(mA=44, mB=28, eA=195.2, eB=71.4, sA=3.941, sB=3.798, T=590, p=1), 2) == 0.52)


# Fuller, et al.
Df = function(mA, mB, VA, VB, T=T0, p=p0) {
	mAB <- 2 / (1/mA + 1/mB)
	return <- 0.00143*T^1.75 / (p*sqrt(mAB)*(VA^(1/3) + VB^(1/3))^2)

}

# Example 11-3: Estimate the diffusion coefficient of allyl chloride (AC) in air at 298 K and 1 bar.
# Result: 0.096 cm²/s
stopifnot(round(Df(mA=29, mB=76.5, VA=19.7, VB=80.25, T=298, p=1), 3) == 0.096)

