#! /bin/bash

# Check arguments...
if [ $# -ne 9 ] ; then
    echo
    echo "============================================================"
    echo "$0"
    echo "============================================================"
    echo
    echo "This script can be used to estimate stratospheric lifetimes"
    echo "of CFC-11 and CFC-12 from zonal mean climatologies."
    echo
    echo "Usage: $0 <clim> <year> <latmin1> <latmax1> "
    echo "          <latmin2> <latmax2> <zmin> <zmax> <tc>"
    echo
    echo "Use <clim> to specify the climatology data file."
    echo "Specify the reference <year>, one or two latitude ranges"
    echo "with [<latmin12>, <latmax12>], and the altitude range"
    echo "with [<zmin>, <zmax>]. Set <tc> to 'y' or 'n' to apply"
    echo "or skip transience correction."
    echo 
    echo "The climatology is required as an ASCII table file:"
    echo "\$1 = latitude [deg]"
    echo "\$2 = altitude [km]"
    echo "\$3 = CFC-11 vmr [ppt]"
    echo "\$4 = CFC-12 vmr [ppt]"
    echo
    echo "Output will be written to stdout and into the working"
    echo "directory. The file 'lifetime.log' summarizes the results"
    echo "of the lifetime analysis. Additional plots are found in"
    echo "'data_f11.ps', 'data_f12.ps', 'corr.ps', and 'slopes.ps'."
    echo
    exit
fi

# Set variables...
clim=$1
year=$2
latmin1=$3
latmax1=$4
latmin2=$5
latmax2=$6
zmin=$7
zmax=$8
tc=$9

# Set surface vmrs (NOAA/ESRL)...
case "$year" in
    1990) f11=261.899 ; f12=481.498 ;;
    1991) f11=266.110 ; f12=493.349 ;;
    1992) f11=269.297 ; f12=504.743 ;;
    1993) f11=270.493 ; f12=511.632 ;;
    1994) f11=270.651 ; f12=517.690 ;;
    1995) f11=269.887 ; f12=524.522 ;;
    1996) f11=269.150 ; f12=529.434 ;;
    1997) f11=267.915 ; f12=533.363 ;;
    1998) f11=266.216 ; f12=536.553 ;;
    1999) f11=264.769 ; f12=539.451 ;;
    2000) f11=262.842 ; f12=541.627 ;;
    2001) f11=260.971 ; f12=542.427 ;;
    2002) f11=258.973 ; f12=543.292 ;;
    2003) f11=256.279 ; f12=542.828 ;;
    2004) f11=253.815 ; f12=542.461 ;;
    2005) f11=251.556 ; f12=541.563 ;;
    2006) f11=249.445 ; f12=540.205 ;;
    2007) f11=246.973 ; f12=537.816 ;;
    2008) f11=244.746 ; f12=535.360 ;;
    2009) f11=242.689 ; f12=532.823 ;;
    2010) f11=240.690 ; f12=530.010 ;;
    *) echo "Reference year not supported!" ; exit
esac

# Filter data...
filt=$(basename $clim).filt
sort -rgk1 -rgk2 $clim | awk -v latmin1=$latmin1 -v latmax1=$latmax1 -v latmin2=$latmin2 -v latmax2=$latmax2 -v zmin=$zmin -v zmax=$zmax '{
  if(NR>0 && $1!="#" && (($1>=latmin1 && $1<=latmax1) || ($1>=latmin2 && $1<=latmax2)) && $2>=zmin && $2<=zmax) {
    if($1!=old1)
      print ""
    print
    old1=$1
  }
}' > $filt

# Plot profiles and correlations...
gnuplot >& /dev/null <<EOF
set term pos enh col sol "Helvetica" 20
set mxtics
set mytics

set out "data_f11.ps"
set xla "CFC-11 vmr [ppt]"
set xra [0:300]
set yla "altitude [km]"
set yra [8:32]
plot "$clim" u 3:2 w lp pt 7 lt 7 lw 2 lc rgbcolor "gray50" t "", \
     "$filt" u 3:2 w lp pt 7 lt 1 lw 2 lc rgbcolor "black" t ""

set out "data_f12.ps"
set xla "CFC-12 vmr [ppt]"
set xra [0:600]
plot "$clim" u 4:2 w lp pt 7 lt 7 lw 2 lc rgbcolor "gray50" t "", \
     "$filt" u 4:2 w lp pt 7 lt 1 lw 2 lc rgbcolor "black"  t ""

set out "corr.ps"
set xla "CFC-11 vmr [ppt]"
set xra [0:300]
set yla "CFC-12 vmr [ppt]"
set yra [0:600]
plot "$clim" u 3:4 w p pt 7 lt 7 lw 2 lc rgbcolor "gray50" t "", \
     "$filt" u 3:4 w p pt 7 lt 1 lw 2 lc rgbcolor "black"  t ""
EOF

# Get slopes for windows...
rm -f slopes.tab
for x in $(seq 50 5 250) ; do
    rm -f fit.log
    gnuplot >& /dev/null <<EOF
f(x)=a*x+b
fit f(x) "$filt" u (\$3>=$x-50 && \$3<=$x+50 ? \$3 : 1/0):4 via a,b
EOF
    m=$(awk -v x=$x '{if($3>=x-50 && $3<=x+50){m+=$3; n++}}END{if(n>0) print m/n, n}' $filt)
    s=$(awk '{if($1=="a" && $2=="=") print $3, $5}' fit.log | tail -n 1)
    echo "$x $m $s" >> slopes.tab
done

# Get slope at tropopause...
rm -f fit.log
gnuplot >& /dev/null <<EOF
f(x)=a*x+b
fit f(x) "slopes.tab" u (\$2>=100 && \$2<=$f11 ? \$2 : 1/0):4 via a,b
EOF
awk -v f11=$f11 '{
  if($1=="a" && $2=="=") {a=$3; da=$5}
  if($1=="b" && $2=="=") {b=$3; db=$5}
}END{
  gt=a*f11+b
  dgt=sqrt((da*f11)^2+db^2)
  print a, da, b, db, gt, dgt
}' fit.log > tropo.tab
a=$(awk '{print $1}' tropo.tab)
b=$(awk '{print $3}' tropo.tab)
gt=$(awk '{print $5}' tropo.tab)
dgt=$(awk '{print $6}' tropo.tab)

# Plot slopes...
gnuplot <<EOF
set term pos enh col sol "Helvetica" 20
set out "slopes.ps"
set mxtics
set mytics
set xla "CFC-11 vmr [ppt]"
set yla "slope (CFC-12/CFC-11)"
set xra [0:300]
set yra [0:3]
plot "slopes.tab" u 2:4:5 w yerr lt 1 lw 2 pt 7 lc rgbcolor "black" t "", \
     (x>=50 && x<=$f11) ? $a*x+$b : 1/0 w l lt 2 lw 2 lc rgbcolor "red" t "", \
     "tropo.tab" u ($f11):5:6 w yerr lw 2 pt 7 lc rgbcolor "red" t ""
EOF

# Get range of data...
ls="$(awk 'BEGIN{min=1e100; max=-1e100}{if(NF>0 && 1.*$1>max) max=1.*$1; if(NF>0 && 1.*$1<min) min=1.*$1}END{print min,"...",max,"deg"}' $filt)"
zs="$(awk 'BEGIN{min=1e100; max=-1e100}{if(NF>0 && 1.*$2>max) max=1.*$2; if(NF>0 && 1.*$2<min) min=1.*$2}END{print min,"...",max,"km"}' $filt)"

# Get lifetime...
echo | awk -v clim="$clim" -v year=$year -v ls="$ls" -v zs="$zs" -v f11=$f11 -v f12=$f12 -v g=$gt -v dg=$dgt -v tc=$tc '{
  
  # Write info...
  print "\nclimatology:", clim
  print "reference year:", year
  print "latitude range:", ls
  print "altitude range:", zs
  
  # Set errors of surface vmrs...
  df11=0.03*f11
  df12=0.03*f12
  printf("CFC-11 surface vmr: %.1f +/- %.1f ppt (%.1f%%)\n", f11, df11, 100*df11/f11)
  printf("CFC-12 surface vmr: %.1f +/- %.1f ppt (%.1f%%)\n", f12, df12, 100*df12/f12)
  
  # Calculate ratio of burdens...
  alpha=0.97
  dalpha=0.01
  r=alpha*f11/f12
  dr=sqrt((dalpha*f11/f12)^2+(alpha*df11/f12)^2+(alpha*df12*f11/f12^2)^2)
  printf("ratio of burdens (CFC-11/CFC-12): %.3f +/- %.3f (%.1f%%)\n", r, dr, 100*dr/r)
    
  # Add systematic error to slope...
  dg=sqrt(dg^2+(0.1*sqrt(2.0)*g)^2.0)
  printf("correlation slope (CFC-12/CFC-11): %.2f +/- %.2f (%.1f%%)\n", g, dg, 100*dg/g)
  
  # Set transience correction factor...
  if(tc=="y") {
    C=0.0163399*year-31.7831
    dC=0.05*C
  } else {
    C=1.0
    dC=0.0
  }
  printf("correction factor for transience (CFC-12/CFC-11): %.3f +/- %.3f (%.1f%%)\n", C, dC, 100*dC/C)
  
  # Get ratio of lifetimes...
  rgc=r*g*C
  drgc=sqrt((dr*g*C)^2+(r*dg*C)^2+(r*g*dC)^2)
  printf("CFC-11/CFC-12 lifetime ratio: %.3f +/- %.3f (%.1f%%)\n", rgc, drgc, 100*drgc/rgc)
  
  # Get lifetime of CFC-12...
  t11=52
  t12=t11/rgc
  dt12m=t11/(rgc-drgc)
  dt12p=t11/(rgc+drgc)
  printf("CFC-12 lifetime (based on %g years for CFC-11): %.1f (%.1f - %.1f) years\n", t11, t12, dt12p, dt12m)
  
}' | tee lifetime.log

# Create jpg...
ps2raster -P -A data_f11.ps data_f12.ps corr.ps slopes.ps
