
PRO event_block, event

;;********************************************************************************
;;                                                                              ;;
;;       Program aiming to plot the average of a select number of               ;;
;;       FD-events for any parameter recorded by the MODIS sattelite            ;;
;;       program. This via a GUI.                                               ;;
;;                                                         January.2012 - JSV   ;;
;;                                                                              ;;
;;********************************************************************************


;*********************************************************************************
;
;      		EVENT-HANDLER - In case of event, do the following:              ;
;
;*********************************************************************************

   Widget_Control, event.top,    get_uvalue = info
   Widget_Control, info.DateID,  get_value  = dates
   Widget_Control, info.listID,  get_uvalue = list
   Widget_Control, info.maskID,  get_value  = mask
   Widget_Control, info.startID, get_value  = startd
   Widget_Control, info.endID,   get_value  = endd
   Widget_Control, info.nanID,   get_value  = nan
   para = widget_info(info.listID, /LIST_SELECT)
   nan2 = widget_info(info.nan2ID, /DROPLIST_SELECT)
   cd,info.dir

; Help button
   if info.helpID eq event.id then begin
      helpstr = [" PROGRAM DESCRIPTION:",+$
                 "This program shows composites of FDs over a selected time",+$ 
                 "interval. Either individual MODIS cloud parameters or the",+$
                 "first principal component of all 6 parameters can be chosen.",+$
                 " ",+$
                 " 1-SIGMA ERROR:",+$
                 "The 1-sigma error is calculated in 2 ways. If time",+$
                 "interval IS [-15,20], the sigma error is taken from file",+$
                 "'std.dat' and shifted along the y-axis to the mean value of",+$
                 "day [-15,-5]. For the PC, this only applies if 5 FDs are",+$
                 "chosen. These errors are calculated via a monte carlo method",+$
                 "as described in the article, and should be regarded as a",+$ 
                 "better estimate. If time interval IS NOT [-15,20], the sigma",+$
                 "error is computed from the standard deviation of the",+$
                 "displayed composite from start date to day 0. For small",+$
                 "time intervals this error estimate may be misleading.",+$
                 " ",+$
                 " PRINCIPAL COMPONENT:",+$
                 "The first principal component is calculated for all 6",+$
                 "parameters via 'PCA.pro'. Note that all NaN's are set to 0,",+$
                 "as 'PCA.pro' cannot handle NaN's.",+$
                 " ",+$
                 " OUTLIERS:",+$
                 "A list of outliers, read from file 'outliers.dat', can be",+$
                 "set to NaN.",+$
                 "",+$
                 " TREATMENT OF NaN's",+$
                 "Two ways of processing NaN's are selectable: Either NaN's",+$
                 "are taken into account as the composite average is",+$
                 "generated ('corrected locally'), or they are allowed to",+$
                 "propagate through to the composite ('corrected globally').",+$
                 "The latter is useful to see where NaN's are present in a",+$ 
                 "composite.",+$
                 "",+$
                 " CONTACT:",+$
                 "For questions or bugs, contact 'jacobsvensmark@gmail.com'."]

      msg = Dialog_Message(helpstr, /CENTER, /INFORMATION) 
      RETURN
   endif

; If no dates are selected, do nothing
   IF total(dates) EQ 0 THEN BEGIN
     plot,[0,0],[0,0],/Nodata
     RETURN
   ENDIF

 data = info.data_nomask
; Set ouliers from 'data/outliers.dat' to NaN
   IF (nan(0)  EQ 1) THEN BEGIN
     dump = ''
     nl   = FILE_LINES('outliers.dat') - 6
     outl = intarr(2,nl)
     openr,1,'outliers.dat'
       for i=0,5 do readf,1,dump
       readf,1,outl
     close,1
     for i=0,nl-1 do data(*,outl(0,i)-1,outl(1,i)+100) = !values.F_nan
   ENDIF
; Misc settings
   p_strings  = [ 'Cloud Effective Emissivity Mean'              ,$
                  'Cloud Condensation Nuclei Ocean Mean'         ,$
                  'Cloud Optical Thickness Liquid Mean'          ,$
                  'Cloud Water Path Liquid Mean'                 ,$
                  'Cloud Fraction Liquid'                        ,$
                  'Cloud Effective Radius Liquid Mean'           ,$ 
                  'First Principal Component']
   unit       = [ ' / Unitless'   ,$
                  ' / cm!u-2!n'   ,$
                  ' / Unitless'   ,$
                  ' / g m!u-2!n'  ,$
                  ' / Unitless'   ,$
                  ' / 10!u-6!n m' ,$ 
                  ' / Unitless']

   d_strings  = [ '31-10-2003'  ,$
                  '19-1-2005'   ,$
                  '13-9-2005'   ,$
                  '16-7-2000'   ,$
                  '12-4-2001'   ,$
                  '10-11-2004'  ,$
                  '26-9-2001'   ,$
                  '17-7-2005']

    close,/all
    ndays   = abs(endd)+abs(startd)+1
    n_e     = total(dates)
    dat     = fltarr(n_e,ndays)
    day     = findgen(ndays)+startd + 0.5
    dump    = ''
    counter = 0

; Get STD's for special case day [-15;20] 
   deviation_array = fltarr(n_elements(p_strings)-1,13)
   IF (startd EQ -15) AND (endd EQ 20) THEN BEGIN 
     openr,1, 'std.dat' & readf,1,dump
       readf,1, deviation_array
     close,1
   ENDIF
; Open selected FD-datafiles and create average for the plot

    ind  = where(dates eq 1)
    daym = where(day eq min(day))
    daya = where(day eq max(day)) 
    day0 = where(day eq 0.5)

  IF para NE 6 THEN BEGIN  ; CREATE MEANS FOR SELECTED PARAMETER (NOT PCA-1)

    dat   = data(para,ind,(100+startd):(100+endd))

    IF (nan2 EQ 1) THEN BEGIN    ; CORRECT NaN's GLOBALLY
      sum = total(dat,2)/n_e
    ENDIF ELSE IF (nan2 EQ 0) THEN BEGIN    ; CORRECT NaNS LOCALLY
      dcount = intarr(ndays)
      for j=0,n_e-1 do begin                           ; 
        c = where(finite(dat(0,j,*),/NAN),count)       ; 
        if count ne 0 then begin                       ;
          dcount(c) = dcount(c)+1                      ;
        endif                                          ;
      endfor                
      sum = total(dat,2,/NAN) 
      for j=0,ndays-1 do sum(j) = sum(j)/(n_e-dcount(j))
    ENDIF

  ENDIF ELSE IF (para EQ 6) THEN BEGIN  ; CREATE MEANS FOR PCA-1

    dat  = data(0:5,ind,(100+startd):(100+endd))      ; 
    for i=0,5 do begin                                ; 
     for j=0,n_e-1 do begin                           ; Remove linear 
      c = where(finite(dat(i,j,*)))                   ; trends from
      r = linfit(day(c),dat(i,j,c))                   ; individual dates
      dat(i,j,c) = dat(i,j,c) - day(c) * r(1) - r(0)  ; and parameters
     endfor                                           ;
    endfor                                            ;

    IF (nan2 EQ 1) THEN BEGIN    ; CORRECT NaN's GLOBALLY
      rdat = total(dat,2)/n_e
      for i=0,5 do rdat(i,*) = rdat(i,*) / stddev(rdat(i,*),/nan)
      testnan = total(WHERE(FINITE(rdat, /NAN)))
      if (testnan ne -1) then begin  ; Set NAN's eq 0
        rdat(WHERE(FINITE(rdat, /NAN))) = 0.0 
      endif
    ENDIF
 
    IF (nan2 EQ 0) THEN BEGIN    ; CORRECT NaNS LOCALLY
    dcount = intarr(6,ndays) 
        for i=0,5 do begin                                ; 
         for j=0,n_e-1 do begin                           ; 
          c = where(finite(dat(i,j,*),/NAN),count)        ; 
          if count ne 0 then begin                        ;
            dat(i,j,c) = 0.0                              ;
            dcount(i,c) = dcount(i,c)+1                   ;
          endif                                           ;
         endfor                                           ;
        endfor                                            ;     
      rdat = total(dat,2) 
      for i=0,5 do for j=0,ndays-1 do rdat(i,j) = rdat(i,j)/(n_e-dcount(i,j))
      for i=0,5 do rdat(i,*) = rdat(i,*) / stddev(rdat(i,*),/nan)
    ENDIF

    PCA, rdat(*,*), EVALUE=evalue, EVECTOR=evector, PC=pc
    sum    = fltarr(ndays)
    sum(*) = pc(*,0)
    sum(*) = sum(*) / stddev(sum(daym:day0))
    dev    = 1.0
    meanv  = 0.0
  ENDIF

; Subtract mean or trend

   if (para ne 6) then begin
      if (startd eq -15) and (endd eq 20) then begin
         dev = deviation_array(para,n_e-1)
         c = where(finite(sum(*)))
         r = linfit(day(c),sum(c))
         if mask(0) eq 1 then begin  
            sum(c) = sum(c) - day(c) * r(1)
            meanv  = mean(sum(daym:day0),/NAN)
         endif
         if (mask(1) eq 1) then begin
            sum = sum - mean(sum(0:9)) ;
            meanv = 0
         endif else meanv = mean(sum(0:9)) 
      endif else begin
         dev = stddev(sum(daym:day0),/NAN)
         c = where(finite(sum(*)))
         r = linfit(day(c),sum(c))
         if mask(0) eq 1 then begin  
            sum(c) = sum(c) - day(c) * r(1)
            meanv  = mean(sum(daym:day0),/NAN)
         endif else meanv = mean(sum(daym:day0))
         if mask(1) eq 1 then begin 
            sum = sum - mean(sum(daym:day0),/NAN)
            meanv  = 0
         endif
      endelse
   endif

   if (para eq 6) and (startd eq -15) and (endd eq 20) and (n_e eq 5) then begin 
      sum = (sum-mean(sum(0:9))) / 1.45835 ; Sigma-error for 5-event composite
   endif                                   ; of day [-100;0]
 
; Plot composite FD
 if (min(sum)-meanv lt -2*dev) then ymin = min(sum)-0.2*dev else ymin = meanv-2.2*dev
 if (max(sum)-meanv lt  2*dev) then ymax = meanv+2.2*dev    else ymax = max(sum)+0.2*dev 

    plot, day, sum(*),/nodata,$;/YNOZERO,       $
        xtitle = "Time / # day",              $
        ytitle = p_strings(para)+unit(para),  $
        title  = p_strings(para),             $
        xrange = [startd-0.5,endd+1.5], xstyle=1, $
        yrange = [ymin,ymax],ystyle=1

    if mask(2) eq 1 then begin
       POLYFILL, [ startd,  endd, endd, startd, startd]+0.5, $
                 [meanv-2*dev, meanv-2*dev, meanv+2*dev, meanv+2*dev, meanv-2*dev], /DATA, $
                 COLOR=0.5 * !D.N_COLORS
       POLYFILL, [ startd,  endd, endd, startd, startd]+0.5, $
                 [meanv-dev, meanv-dev, meanv+dev, meanv+dev, meanv-dev], /DATA, $
                 COLOR=0.75 * !D.N_COLORS
       oplot, [startd,endd]+0.5,[meanv,meanv], linestyle=4
    endif

    oplot, day, sum(*)

  if info.saveID eq event.id then begin
     saveimage, p_strings(para) + '.jpg',/jpeg,quality=100
  endif
if info.valuID eq event.id then begin 
  for i=0,n_elements(day)-1 do print,day(i),sum(i)
endif
END


;********************************************************************************

;		WIDGET BASE GUI - Here is how the buttons are arranged		;

;********************************************************************************


PRO FD_explore

; SET PATH MANUALLY HERE TO DIRECTORY CONTAINING "FD_explore.pro" and data
; Base directory
  dir = '/path/path/path'
  cd,dir

; Placement and hierarchy of buttons:

  base0     = widget_base (title= 'Controls',/row)
  base_LHS  = widget_base (base0, /column)
  i         = widget_label(base_LHS, value='Parameter plotted:' )
  base_list = widget_base (base_LHS)
  i         = widget_label(base_LHS, value='Removal of linear trend:' )
  base_geog = widget_base (base_LHS, /row)
  i         = widget_label(base_LHS, value='FD dates included:' )
  base_butt = widget_base (base_LHS, column=2)
  i         = widget_label(base_LHS, value='Time interval (between [-100,100]):')
  base_intv = widget_base (base_LHS, column=2) 
  i         = widget_label(base_LHS, value="Treatment of outliers and NaN's:" )
  base_nan  = widget_base (base_LHS, column=1) 
  base_plot = widget_base (title='Plotting Window')
  grid      = widget_base (base_LHS,column=3)

; FD-date Buttons

  date       = ['#01: 31-10-2003',$
                '#02: 19-01-2005',$
                '#03: 13-09-2005',$
                '#04: 16-07-2000',$
                '#05: 12-04-2001',$
                '#06: 10-11-2004']

  DateID     = CW_BGROUP(base_butt, date, BUTTON_UVALUE=dateu, column=2, /NONEXCLUSIVE, SET_VALUE=[1,1,1,1,1,0])

; MODIS Parameter list

  parameters = ['Effective Emissivity'      ,$
                'Condensation Nuclei Ocean               ' ,$
                'Optical Thickness Liquid'  ,$
                'Water Path Liquid'         ,$
                'Cloud Fraction Liquid'     ,$
                'Effective Radius Liquid'   ,$
                'First Principal Component']

  options    = ['Remove slope'              ,$
                'Remove avg.'               ,$
                'Sigma error']

  p_str      = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14]
  listID     = Widget_List (base_list, YSize=7,  Value=parameters, UValue=p_str)
  Widget_Control, listID, SET_LIST_SELECT=0

; Geographical focus point

  maskID     = CW_BGROUP(base_geog, options, BUTTON_UVALUE=['1','1','1'], /NONEXCLUSIVE, SET_VALUE=[0,0,1],column=2)
  startID    = CW_FIELD(base_intv, /integer, title="Start: ",value=-15,xsize=4)
  endID      = CW_FIELD(base_intv, /integer, title="  End: ",value= 20,xsize=4)
  nanID      = CW_BGROUP(base_nan,['Outliers=NaN'],/NONEXCLUSIVE, BUTTON_UVALUE=['1'], SET_VALUE=[1],row=1)
  nan2ID     = WIDGET_DROPLIST(base_nan,value=['Correct NaNs locally','Correct NaNs globally'],uvalue=[1,2])
; Plotting area

  plotID     = Widget_Draw (base_PLOT,  Xsize=555, Ysize=400)

; Save and help and print button
  
   saveID = WIDGET_BUTTON (grid, VALUE = 'Save window')
   valuID = WIDGET_BUTTON (grid, VALUE = 'Print values')
   helpID = WIDGET_BUTTON (grid, VALUE = 'Help...')

; Load data

  N1    = 0
  N2    = 0
  N3    = 0

  openr,1, 'MODIS_global_means.dat'
     readf,1,N1,N2,N3
     data_nomask = fltarr(N1,N2,N3)
     readf,1,data_nomask
  close,1

  dims = [N1,N2,N3]

; Realizing GUI

  Widget_Control, base0,  /realize

; Info-structure to send ID's to the event-handler

  info = { listID:listID           ,$
           dateID:dateID           ,$
           plotID:plotID           ,$
           maskID:maskID           ,$
           nanID:nanID             ,$
           saveID:saveID           ,$
           data_nomask:data_nomask ,$
           dims:dims               ,$
           startID:startID         ,$ 
           endID:endID             ,$
           valuID:valuID           ,$
           nan2ID:nan2ID           ,$
           dir:dir                 ,$
           helpID:helpID            $
           }

  Widget_Control, base0, Set_UValue=info
  xmanager, 'WHATEVER', base0, EVENT_HANDLER='event_block'

END
