      subroutine INIT_RANDOM_SEED()
      implicit none
      integer :: i, n, clock
      integer, dimension(:), allocatable :: seed
      call RANDOM_SEED(size = n)
      allocate(seed(n))
      call SYSTEM_CLOCK(COUNT=clock)
      seed = clock + 37 * (/ (i - 1, i = 1, n) /)
      call RANDOM_SEED(PUT = seed)
      deallocate(seed)
      end subroutine INIT_RANDOM_SEED

! ======================================================================

      subroutine TEST_ADJ(lrandom,tol_roundoff)
! ----------------------------------------------------------------------
! This routine makes the test of the adjoint, i.e.,
!     <x,LXs>=<L^Tx,Xs>
! where:
!     x     is a (nlon,nlat,nlev) vector defined on the lat/lon/lev grid
!     Xs    is a (0:NX,nlev) spectral representation of a physical field
!     L     is the operator that links the analysis increment on the LL
!           grid (x-xb) and in the spectral space Xs
! If <x,LXs>/<L^Tx,Xs> > tol_roundoff*EPSILON(1.d0), then the test fails.
! tol_roundoff is set to 2.d2 and used to account for the roundoff error. 
! This value might be sligthly adapted to the computers.
!
! Inputs:
!  lrandom: If .true., Xs, x_sg and x_mg are random variables. If .false., 
!           Xs, x_sg and x_mg are set to fixed values. This latest mode
!           allows to reproduce the same values for the test of the adjoint
!           and is useful for debugging.
!  tol_roundoff:   The tolerance value for round off error.
!
! Authors: quentin@oma.be
! Date:     Nov 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlat, nlon, nlev, xlon, ylat
      use BGLIB, only : nmu, nlat_sg, nmax, Ns, NX, lambda, Pnm, wg, mu, lprecomp_bglib
      implicit none
      logical, intent(in) :: lrandom
      real*8, intent(in) ::  tol_roundoff

      integer :: k, j, i, ix, iy, i1, i2, l, PnmIndex
      integer :: idx, n, m, nn, mm, iP
      real*8 :: x1,x2,x3,dlat, dlon, nn1
      real*8 :: xv(nlon*nlat_sg*nlev), SXv(nlon*nlat_sg*nlev)

      real*8 , dimension(nlon,nlat_sg,nlev) :: x_sg, xx_sg
      real*8 , dimension(nlon,nlat,nlev) :: x_mg, xx_mg
      complex*16 , dimension(NX) :: Xstmp
      complex*16 , dimension(NX,nlev) :: Xs, XXs
      real*8 , dimension(NX,nlev) :: rXs, iXs
      integer, parameter :: nn2 = nlon*nlat*nlev
      real*8 :: xn1(nmax), xn2(nmax)

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling S_DIRECT'
        print*,'Fatal error in S_DIRECT'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! 1. Make initial state of Xs,x_sg,x_mg
! ----------------------------------------------------------------------
      if (lrandom) then
        call INIT_RANDOM_SEED()
        call RANDOM_NUMBER(rXs)
        call RANDOM_NUMBER(iXs)
        Xs = CMPLX(rXs,iXs)
        print*,'kiki'
        do l=1,nlev
          call S_DIRECT(.false.,Xs(:,l),x_sg(:,:,l))
          print*,'tutu'
          call SG2MG(x_sg(:,:,l),x_mg(:,:,l))
        enddo
        print*,'coucou'
        call RANDOM_NUMBER(rXs)
        call RANDOM_NUMBER(iXs)
        Xs = CMPLX(rXs,iXs)
        do l=1,nlev
          call S_DIRECT(.false.,Xs(:,l),x_sg(:,:,l))
        enddo
        call RANDOM_NUMBER(rXs)
        call RANDOM_NUMBER(iXs)
        Xs = CMPLX(rXs,iXs)
      else
        do l=1,nlev
          do n=0,Ns
          do m=0,n
            iP = PnmIndex(n,m)
!            iP = PlmIndex(n,m)
            Xs(iP,l) = CMPLX(iP*l*1d-10,2*iP*l*1d-10)
            Xstmp(iP) = CMPLX(3*iP*l*1d-10,4*iP*l*1d-10)
          enddo
          enddo
          call S_DIRECT(Xstmp,x_sg(:,:,l))
          call SG2MG(x_sg(:,:,l),x_mg(:,:,l))
        enddo
      endif

! ----------------------------------------------------------------------
! 1. Start message
! ----------------------------------------------------------------------
      print*,'In TEST_ADJ.'
      print*,'  In the following, x stand for the model state vector on the'
      print*,'  grid and Xs stand for its spectral representation.'
      print*,'  S represents the spectral transform operator.'
      print*,'  S* represents the adjoint of the spectral transform operator.'
      print*,'  C represents the square root of the correlation matrix.'
      print*,'  G represents the matrix of transformation from the spectral grid'
      print*,'    to the model grid.'
      print*,'  G^T represents the transpose matrix of G.'
      print*,'  E represents the background error standard deviation matrix.'
      
! ----------------------------------------------------------------------
! 2. Test of spectral transform
! ----------------------------------------------------------------------
! --- Oper L on Xs ---
      do l=1,nlev
        call S_DIRECT( Xs(:,l) , xx_sg(:,:,l) )
      enddo
      x1 = DOT_PRODUCT( RESHAPE( x_sg, (/nlon*nlat_sg*nlev/) ) , RESHAPE( xx_sg, (/nlon*nlat_sg*nlev/) ) )

! --- Oper L^T on x ---
      do l=1,nlev
        call S_ADJ( x_sg(:,:,l), XXs(:,l) )
      enddo
      call DOT_PROD_SPECTRAL(Xs,XXs,x2)

! --- Print results ---
      print*, 'results of <x,SXs> ?= <S*x,Xs>'
      print*, '  <x,SXs> =          ',x1
      print*, '  <S*x,Xs> =         ',x2
      print*, '  <x,SXs>/<S*x,Xs> = ',x1/x2
      if (abs(1d0-x1/x2)>tol_roundoff*EPSILON(1.d0)) then
        print*,' Test LL* failed'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! 3. Test of spectral transform and spatial correlations
! ----------------------------------------------------------------------
! --- Oper L^T on x ---
      do l=1,nlev
        call S_ADJ( x_sg(:,:,l), XXs(:,l) )
      enddo
      call SQRT_CORREL(.false.,1,XXs)
      call DOT_PROD_SPECTRAL(Xs,XXs,x2)

! --- Oper L on Xs ---
      XXs = Xs
      call SQRT_CORREL(.true.,1,XXs)
      do l=1,nlev
        call S_DIRECT( XXs(:,l) , xx_sg(:,:,l) )
      enddo
      x1 = DOT_PRODUCT( RESHAPE( x_sg, (/nlon*nlat_sg*nlev/) ) , RESHAPE( xx_sg, (/nlon*nlat_sg*nlev/) ) )

! --- Print results ---
      print*, 'results of <x,SCXs> ?= <CS*x,Xs>'
      print*, '  <x,SCXs> =           ',x1
      print*, '  <CS*x,Xs> =          ',x2
      print*, '  <x,SCXs>/<CS*x,Xs> = ',x1/x2
      if (abs(1d0-x1/x2)>tol_roundoff*EPSILON(1.d0)) then
        print*,' Test LL* failed'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! 4. Test of spectral transform, spatial correlations and grid transform.
! ----------------------------------------------------------------------
! --- Oper L^T on x_mg ---
      do l=1,nlev
        call AD_SG2MG(x_mg(:,:,l),xx_sg(:,:,l))
        call S_ADJ( xx_sg(:,:,l), XXs(:,l) )
      enddo
      call SQRT_CORREL(.false.,1,XXs)
      call DOT_PROD_SPECTRAL(Xs,XXs,x2)

! --- Oper L on Xs ---
      XXs = Xs
      call SQRT_CORREL(.true.,1,XXs)
      do l=1,nlev
        call S_DIRECT( XXs(:,l) , xx_sg(:,:,l) )
        call SG2MG(xx_sg(:,:,l),xx_mg(:,:,l))
      enddo
      x1 = DOT_PRODUCT( RESHAPE( xx_mg, (/nlon*nlat*nlev/) ) , RESHAPE( x_mg, (/nlon*nlat*nlev/) ) )

! --- Print results ---
      print*, 'results of <x,GSCXs> ?= <CS*G^Tx,Xs>'
      print*, '  <x,GSCXs> =              ',x1
      print*, '  <CS*G^Tx,Xs> =           ',x2
      print*, '  <x,GSCXs>/<CS*G^Tx,Xs> = ',x1/x2
      if (abs(1d0-x1/x2)>tol_roundoff*EPSILON(1.d0)) then
        print*,' Test LL* failed'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! 5. Test of spectral transform, spatial correlations, grid transform 
!    and standard deviation of B.
! ----------------------------------------------------------------------
! --- Oper L^T on x_mg ---
      xx_mg = x_mg
      do l=1,nlev
        call B_STDDEV(l, 1, xx_mg(:,:,l) )
        call AD_SG2MG(xx_mg(:,:,l),xx_sg(:,:,l))
        call S_ADJ( xx_sg(:,:,l), XXs(:,l) )
      enddo
      call SQRT_CORREL(.false.,1,XXs)
      call DOT_PROD_SPECTRAL(Xs,XXs,x2)

! --- Oper L on Xs ---
      XXs = Xs
      call SQRT_CORREL(.true.,1,XXs)
      do l=1,nlev
        call S_DIRECT( XXs(:,l) , xx_sg(:,:,l) )
        call SG2MG(xx_sg(:,:,l),xx_mg(:,:,l))
        call B_STDDEV(l, 1, xx_mg(:,:,l) )
      enddo
      x1 = DOT_PRODUCT( RESHAPE( xx_mg, (/nlon*nlat*nlev/) ) , RESHAPE( x_mg, (/nlon*nlat*nlev/) ) )

! --- Print results ---
      print*, 'results of <x,EGSCXs> ?= <CS*G^TEx,Xs>'
      print*, '  <x,EGSCXs> =               ',x1
      print*, '  <CS*G^TEx,Xs> =            ',x2
      print*, '  <x,EGSCXs>/<CS*G^TEx,Xs> = ',x1/x2
      if (abs(1d0-x1/x2)>tol_roundoff*EPSILON(1.d0)) then
        print*,' Test LL* failed'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! 6. End
! ----------------------------------------------------------------------
      print*,' '
      print*, 'TEST_ADJ ok'
      print*,' '
      end subroutine TEST_ADJ
