! ======================================================================

      module BGLIB
! ----------------------------------------------------------------------
! Module BGLIB contains all variables and routines that deal with the
! background error covariance matrix of BASCOE. This is explained in more
! details in the document 'bascoe_cov_matrix_vxx.pdf' (today - 5/11/2010 -
! vxx = v06)
! Authors: quentin@oma.be
! Date:     Oct 2010
! ----------------------------------------------------------------------

#if (PRECOMP != 1)
      use SHTOOLS
#endif
      use VAR_DEF, only : nlat, nlon, nlev, nvmr, vmrfg
      integer, parameter :: nmu = nlat-1
      integer, parameter :: Ns = MAX(nlat,nlon/2)-1
      integer, parameter :: NX = (Ns+1)*(Ns+2)/2
      integer :: nlat_sg   ! ie dims of latitude on the physical grid of the spectral transforms
      real*8 :: pi, Lh, Lv
      real*8, allocatable, dimension(:) :: mu, wg, lambda, mu_half, glat, ylat_sg
      real*8, allocatable, dimension(:,:) :: G, G_TRANSP, Pnm, bg_std
      real*8, allocatable, dimension(:,:,:) :: Cv, Ch
      complex*16, allocatable, dimension(:,:) :: Eiml
      logical, save :: lprecomp_bglib = .false., lgg = .false.
      real*8, parameter :: A0 = 6356.7d0            ! effective planet radius (km)
      character(len=2) :: SpectralGrid
      character(len=3) :: bg_units
      character(len=10) :: hcorr_model, vcorr_model
      character(len=10) :: ctmp

      integer, parameter :: nmax = 2*NX*nlev*nvmr

      PRIVATE
      PUBLIC :: PRECOMPUTE_BGLIB
      PUBLIC :: pi, nmu, nmax, nlat_sg, Ns, NX, lgg, A0      &
     &        , mu, wg, ylat_sg, lambda, G, G_TRANSP, Pnm, Eiml    &
     &        , Lh, Lv, hcorr_model, vcorr_model, Cv, Ch, ctmp &
     &        , bg_units, bg_std, lprecomp_bglib, SpectralGrid

      contains

! ======================================================================

      subroutine PRECOMPUTE_BGLIB( ylat )
! ----------------------------------------------------------------------
! This routine precomputes the Gaussian latitudes and weights, the associated 
! Legendre functions, the matrices G and G^T, the spectral coefficients
! of the horizontal correlations and the square root of the vertical correlation
! matrix
! Input:
!   ylat          The latitude grid of the model (lat/lon) grid
!
! Inputs passed via module:
!   SpectralGrid  The type of spectral grid links to the spectral transform.
!                 Possible values are 'gg' (Gaussian grid) or 'll' (lat/lon
!                 grid)
!   hcorr_model   Two horizontal correlation models are accepted: 'gauss'
!                 for Gaussian and 'soar' for second order auto regressive.
!   vcorr_model   Two vertical correlation model requires: either 'hat' or
!                 'gauss_lev'.
!
! Authors: quentin@oma.be
! Date:     Feb 2012
! ----------------------------------------------------------------------

      implicit none
      real*8, intent(in), dimension(nlat) :: ylat
      integer :: status(18), imu, imu1(1), imu2(1), nn
      integer :: k, j, n, l, m, i, idx, len, ios, ilev, ilat, ivmr, nv
      integer :: nlev_in, ilx(1), il1, il2, i1, i2, i3, i4, PnmIndex
      real*8, allocatable, dimension(:) :: Lh_in, bg_in, pin
      real*8 :: dlat, d2, xm1, xm2, yll(nlat), dmu(nmu)
      real*8 :: dg2rd, c, q, tmp, pmin, pmax, latmin, latmax
      logical :: ok
      character(len=256) :: flnm

! ----------------------------------------------------------------------
!     Check dims 
! ----------------------------------------------------------------------
#if (PRECOMP==1)
      if (nlat/=60 .and. nlon/=120 .and. nlev/=31 .and. nvmr/=1) then
        print*,'Dimensions of input variable to read in PRECOMPUTE_BGLIB'
        print*,'differ from the dimensions set in var_def.mod.f90.'
        print*,'Only the following dimensions are accepted:'
        print*,'        nlat=60'
        print*,'        nlon=120'
        print*,'        nlev=31'
        print*,'        nvmr=1'
        print*,'If you want to run with different dimensions, re-compile'
        print*,'the code as:'
        print*,'> make BG=precomp'
        print*,'Fatal error in bglib_sh_precomp.f90/PRECOMPUTE_BGLIB.'
      endif
      print*,'SH values read in input file'
#else
      print*,'SH values provided by SHTOOLS'
#endif
      print*,'Dimensions: nlon, nlat, N:',nlon,nlat,Ns

! ----------------------------------------------------------------------
!     Set the dims of the latitude grid on the physical grid
! ----------------------------------------------------------------------
      if (SpectralGrid=='gg') then
        nlat_sg = nmu
        lgg = .true.
      elseif (SpectralGrid=='ll') then
        nlat_sg = nlat
      else
        print*,'Unknown value for SpectralGrid:>'//SpectralGrid//'<'
        print*,'Fatal error in PRECOMPUTE_BGLIB.'
        call EXIT(1)
      endif
!      Ns = nlat_sg-1
!      NX = (Ns+1)*(Ns+2)/2
!      nmax = 2*NX*nlev*nvmr

! ----------------------------------------------------------------------
!     Memory allocation
! ----------------------------------------------------------------------
      status(:) = 0
      ALLOCATE(lambda(nlon),STAT = status(1) )
      ALLOCATE(mu(nmu),STAT = status(2) )
      ALLOCATE(mu_half(nmu+1),STAT = status(3) )
      ALLOCATE(wg(nmu),STAT = status(4) )
      ALLOCATE(ylat_sg(nlat_sg),STAT = status(5) )
      ALLOCATE(G(nlat,nmu),STAT = status(6) )
      ALLOCATE(G_TRANSP(nmu,nlat),STAT = status(7) )
      ALLOCATE(Ch(0:Ns,nlev,nvmr),STAT = status(9) )
      ALLOCATE(Cv(nlev,nlev,nvmr),STAT = status(10) )
      ALLOCATE(Eiml(-Ns:Ns,nlon),STAT = status(11) )
      ALLOCATE(Pnm(nlat_sg,NX),STAT = status(12) )
      if (ANY(status/=0)) then
        print*,'Failed to allocate memory for BG Variables.'
        print*,'Status(:) = ',status
        call EXIT(1)
      else
        print*,'Allocation of BG Variables OK'
      endif

! ----------------------------------------------------------------------
!     Definition of: pi and lambda
! ----------------------------------------------------------------------
      pi=acos(-1.d0)
      dg2rd = pi/180.d0       ! for compatibility with gfortran
      do j=0,nlon-1
        lambda(j+1) = (2*pi)/(nlon)*j
      enddo

! ----------------------------------------------------------------------
!     Set mu and wg
! ----------------------------------------------------------------------
#if (PRECOMP==1)
      open(15,file='../input/gauss_lat_weight.dat')
      read(15,*)ctmp
      do k=1,nmu
        read(15,'(i3,2(2x,es22.15))')i1,mu(k),wg(k)
      enddo
      close(15)
#else
      call PreGLQ ( -1.d0, 1.d0, nmu, mu, wg )
#endif

! ----------------------------------------------------------------------
!     Definition of: ylat_sg
! ----------------------------------------------------------------------
      if (SpectralGrid=='gg') then
        ylat_sg = mu
      elseif (SpectralGrid=='ll') then
        do k=1,nlat_sg
          ylat_sg(k) = sin(ylat(nlat-k+1)*dg2rd)
        enddo
      endif

! ----------------------------------------------------------------------
!     Set associated Legendre functions Pnm
! ----------------------------------------------------------------------
#if (PRECOMP==1)
      open(15,file='../input/assoc_Leg_fun_'//SpectralGrid//'.dat')
      read(15,*)ctmp
      read(15,*)ctmp
      read(15,*)ctmp
      do n=0,Ns
      do m=0,n
        j = PnmIndex(n,m)
        do k=1,nlat_sg
          read(15,'(4i6,2x,es22.15)')i1,i2,i3,i4,Pnm(k,j)
        enddo
      enddo
      enddo
      close(15)
#else
      do k=1,nlat_sg
        call  PlmBar ( Pnm(k,:), Ns, ylat_sg(k), CSPHASE=1, CNORM=1 )
      enddo
      Pnm = Pnm/sqrt(2.d0)
#endif

! ----------------------------------------------------------------------
!     Definition of mu_half (in degree) (Only useful for Gaussian Grid)
! ----------------------------------------------------------------------
      mu_half(1) = 90.d0
      mu_half(nmu+1) = -90.d0
      do k=2,nmu
        mu_half(k)= asin( sin(mu_half(k-1)*dg2rd) - wg(k-1) )/dg2rd     ! for compatibility with gfortran
      enddo

! ----------------------------------------------------------------------
!     Precalculate Exp( (0.d0,1.0D0)*m*lambda ) to optimize the CPU time
!     of S_DIRECT, S_INV and S_ADJ
! ----------------------------------------------------------------------
      do m=-Ns,Ns
      do j=1,nlon
        Eiml(m,j) = exp( (0.d0,1.0D0)*m*lambda(j) )
      enddo
      enddo

! ----------------------------------------------------------------------
!     Definition of matrix G that makes the transformation from the Gaussian 
!     grid to the LL grid.
! ----------------------------------------------------------------------
      dlat = 180.d0/(nlat-1)
      d2 = dlat*0.5d0
      yll = (/ (ylat(k), k=nlat,1,-1) /)
      G(:,:) = 0.d0
      do l=2,nlat-1
      ! find lower/upper index of gaussian grid
        xm1 = MIN(90.d0,yll(l)+d2)
        xm2 = MAX(-90.d0,yll(l)-d2);
        imu1 = MINLOC( ABS(xm1-mu_half) )
        if (mu_half(imu1(1))<xm1) imu1(1) = imu1(1) - 1
        imu2 = MINLOC( ABS(xm2-mu_half) )
        if (mu_half(imu2(1))<xm2) imu2(1) = imu2(1) - 1
      ! Loop on subindex imu and make the weighted average
        idx = 0
        imu = imu1(1)
        do while (imu<=imu2(1))
          idx = idx + 1
          if (imu == imu1(1)) then
            dmu(idx) = xm1 - mu_half(imu+1)
          elseif (imu == imu2(1)) then
            dmu(idx) = mu_half(imu) - xm2
          else
            dmu(idx) = mu_half(imu) - mu_half(imu+1)
          endif
          imu = imu + 1
        enddo
        G(l,imu1(1):imu2(1)) = dmu(1:idx)/dlat
      enddo
      ! Poles
      G(1,1) = dlat*0.5d0
      G(nlat,nmu) = dlat*0.5d0

! ----------------------------------------------------------------------
!     Definition of transpose of G
! ----------------------------------------------------------------------
      G_TRANSP = TRANSPOSE(G)

! ----------------------------------------------------------------------
!     Set the horizontal correlation matrix Cv^(1/2)
!     Two correlation models are accepted: 'gauss' for Gaussian and 'soar'
!     for second order auto regressive.
! ----------------------------------------------------------------------
      Ch(:,:,:) = 0.d0
      do ivmr=1,nvmr
      do ilev=1,nlev
        call HCORR_COEFF(TRIM(ADJUSTL(hcorr_model)),Lh,Ch(:,ilev,ivmr))
      enddo
      enddo

! ----------------------------------------------------------------------
!     Set the square root of the vertical correlation matrix.
! ----------------------------------------------------------------------
#if (PRECOMP==1)
      print*,'Try to read the vertical correlation matrix ...'
      open(15,file='../input/sqrt_vertic_corr.dat')
      read(15,'(16x,i10)')i1
      if (i1/=nlev) then
        print*,' ... Failed. The size of the vertical correlation matrix ' 
        print*,'     differ from (nlev x nlev)'
        print*,' ... nlev=',nlev
        print*,' ... size in file=',i1
        call EXIT(1)
      endif
      read(15,'(20x,a)')ctmp
      if (vcorr_model/=TRIM(ADJUSTL(ctmp))) then
        print*,' ... Failed. The vertical correlation matrix in file does '
        print*,'     not correspond to the requested correlation model'
        print*,' ... requested correlation model=',vcorr_model
        print*,' ... correlation model used to make the correlation matrix=>'//TRIM(ADJUSTL(ctmp)) //'<'
        call EXIT(1)
      endif
      read(15,'(27x,f10.3)')tmp
      if (abs(tmp-Lv)>EPSILON(Lv)) then
        print*,' ... Failed. The vertical correlation matrix in file is '
        print*,'     based on a different correlation length scale than requested.'
        print*,' ... requested correlation length scale=',Lv
        print*,' ... correlation length scale used to make the correlation matrix=',tmp
        call EXIT(1)
      endif
      do i=1,nlev
        read(15,*)i1,Cv(i,:,1)
      enddo
      close(15)
      print*,'... done'
#else
      do ivmr=1,nvmr
         call VCORR_COEFF(vcorr_model,Lv,Cv(:,:,ivmr))
!         open(15,file='../input/sqrt_vertic_corr.dat')
!         write(15,*),'nlev=',nlev
!         write(15,*),'correlation model: ',vcorr_model
!         write(15,*),'correlation length scale: ',Lv
!         do i=1,nlev
!           write(15,'(i3,100(2x,es22.15))')i,Cv(i,:,ivmr)
!         enddo
!         close(15)
      enddo
#endif

! ----------------------------------------------------------------------
!     Set value of lprecomp_bglib
! ----------------------------------------------------------------------
      lprecomp_bglib = .true.

      end subroutine PRECOMPUTE_BGLIB

! ======================================================================

      end module BGLIB

! ======================================================================
! ======================================================================
! ======================================================================

      subroutine HCORR_COEFF(hcorr_model,Lh,Ch)
! ----------------------------------------------------------------------
! This routine calculates the element n of the horizontal correlation
! spectrum as:
!     c_n = C * SUM_k=0^Ns( f(mu_k) * Pn0(mu_k) * W(mu_k) )
!       C = (2/(2*n+1))^(1/2)
!
! where: f(mu_k) = exp( -2*A^2(1-mu_k)/(2*Lh^2)  for Gaussian correlation model
!
!        f(mu_k) = (1 + (sqrt(2*(1-mu_k))/(L/A)) )
!                * exp(-sqrt(2*(1-mu_k))/(L/A))  for SOAR correlation model
! Authors: quentin@oma.be
! Date:     Feb 2012
! ----------------------------------------------------------------------
#if (PRECOMP != 1)
      use SHTOOLS
#else
      use BGLIB, only : ctmp
#endif
      use BGLIB, only : Ns, A0, SpectralGrid
      
      implicit none
      character(len=*), intent(in) :: hcorr_model
      real*8, intent(in) :: Lh
      real*8, dimension(0:Ns), intent(out) :: Ch
      integer :: k, idx, n, i1, i2
      real*8 :: LA, q, qk, Pn(Ns+1,Ns+1)
      real*8 :: mu(Ns+1), wg(Ns+1)
 

      LA = (Lh/A0)
#if (PRECOMP==1)
      open(15,file='../input/Leg_polyn_'//SpectralGrid//'.dat')
      read(15,*)ctmp
      read(15,*)ctmp
      do n=0,Ns
      do k=1,Ns+1
        read(15,*)i1,i2,mu(k),wg(k),Pn(k,n+1)
      enddo
      enddo
      close(15)
#else
      call PreGLQ ( -1.d0, 1.d0, Ns+1, mu, wg )
      do k=1,Ns+1
        call PlBar ( Pn(k,:), Ns, mu(k) )
      enddo
      Pn = Pn/sqrt(2.d0)
!      open(15,file='../input/Leg_polyn_'//SpectralGrid//'.dat')
!      write(15,*)'Ns=',Ns
!      write(15,'(2a6,3(2x,a22))')'n','k','mu(k)','wg(k)','Pn(k)'
!      do n=0,Ns
!      do k=1,Ns+1
!        write(15,'(2i6,3(2x,es22.15))')n,k,mu(k),wg(k),Pn(k,n+1)
!      enddo
!      enddo
!      close(15)
#endif      

      do n=0,Ns
        q = 0.d0
        do k=1,Ns+1
          if (hcorr_model=='gauss') then
            qk = Pn(k,n+1) * wg(k) * exp( -(1.d0-mu(k)) / (LA*LA) )
          elseif (hcorr_model=='soar') then
            qk = Pn(k,n+1) * wg(k) * (1 + (sqrt(2*(1-mu(k)))/LA) )           &
       &      * exp(-sqrt(2*(1-mu(k)))/LA)
          elseif (hcorr_model=='uncorr') then
            if (k==1) then
              qk = Pn(k,n+1) * wg(k)
            else 
              qk = 0
            endif
          else
            print*,'Fatal error in HCORR_COEFF (BGLIB): unknown correlation model:>'//hcorr_model//'<'
            call EXIT(1)
          endif
          if (abs(qk)>EPSILON(qk)) then
            q = q + qk
          endif
!          write(*,'(2i4,10(es15.6,2x))')k,n,Pn(k,n+1),wg(k),mu(k),q 
        enddo
        
!        print*,n,q,q * dsqrt(2.d0/(2.d0*n+1.d0))
        q = q * dsqrt(2.d0/(2.d0*n+1.d0))
        if (q < 0.d0) then
          if (abs(q)<10*EPSILON(q)) then
!            print*,'Warning in HCORR_COEFF, found neg coefficient'
!            print*,n,Lh,q,EPSILON(q)*10
            q = 0.d0
          else
            print*,'Error in HCORR_COEFF, found neg coefficient'
            print*,q,EPSILON(q)*10
            call EXIT(1)
          endif
        endif
        Ch(n) = DSQRT(q)
      enddo

      end subroutine HCORR_COEFF

! ======================================================================

      subroutine VCORR_COEFF(vcorr_model,Lv,C)
! ----------------------------------------------------------------------
! This routine calculates the square root of the vertical correlation 
! matrix. 
! Input:
!  vcorr_model:   the name of the correlation model. Three possibilities:
!                 'identity' (i.e. no vertical correlations), 'hat' and 
!                 'gauss_lev'. 
!  Lv             Vertical correlation length scale. Only used by 'gauss_lev'
!
! Output:
!  Ch             The square root of the vertical correlation matrix.
! 
! A forth correlation model can also be included by de-comment the lines that
! starts by 'elseif (vcorr_model=='gauss_km') then'. It provides the 'gauss_km'
! correlation model. Moreover the correlation length scale must be given in [km].
! For this model, the vertical grid vector 'zstd' (in km) must be given.
! 
! Authors: quentin@oma.be
! Date:     Feb 2012
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlev
      implicit none
      character(len=*), intent(in) :: vcorr_model
      real*8, intent(in) :: Lv
      real*8, dimension(nlev,nlev), intent(out) :: C
      integer :: l,j,info
      real*8 :: d(nlev), e(nlev-1), tau(nlev), z(nlev,nlev)
      real*8 :: work1(nlev), work(2*nlev-2)

      C(:,:) = 0.d0
      if (vcorr_model=='hat') then
        z(:,:) = 0
        d(:) = 1.d0
        e(:) = 0.5d0
        do l=1,nlev
          z(l,l) = 1
        enddo
      elseif (vcorr_model=='identity') then
        z(:,:) = 0
        d(:) = 1.d0
        e(:) = 0.d0
        do l=1,nlev
          z(l,l) = 1
        enddo
!      elseif (vcorr_model=='gauss_km') then
!        do l=1,nlev
!        do j=1,nlev
!            C(l,j) = exp(-0.5*(abs(zstd(l)-zstd(j))/Lv)**2)
!        enddo
!        enddo
!        call SYTRD(C, d )
!        z = C
!        call ORGTR(z, d)
!        do l=1,nlev
!          d(l) = C(l,l)
!          if (l<nlev) e(l) = C(l,l+1)
!        enddo
      elseif (vcorr_model=='gauss_lev') then
        do l=1,nlev
        do j=1,nlev
            C(l,j) = exp(-0.5*(abs(l-j)/Lv)**2)
        enddo
        enddo
! Reduces a the symmetric matrix C to symmetric tridiagonal form T by an 
! orthogonal similarity transformation: A = Q*T*QT. The orthogonal matrix 
! Q is not formed explicitly but is represented as a product of n-1 
! elementary reflectors. Routines are provided for working with Q in this 
! representation (see Application Notes below).
        call DSYTRD('U', nlev, C, nlev, d, e, tau, work, nlev, info)
!        call SYTRD(C, d )
        z = C
        call DORGTR('U', nlev, z, nlev, tau, work, nlev, info)
!        call ORGTR(z, d)
        do l=1,nlev
          d(l) = C(l,l)
          if (l<nlev) e(l) = C(l,l+1)
        enddo
      else
        print*,'Fatal error: Unkown correlation model in VCORR_COEFF'
        print*,'vorr_model=>'//vcorr_model//'<'
        call EXIT(1)
      endif
      call DSTEQR('V', nlev, d, e, z, nlev, work , info)
!     print*,info
      do l=1,nlev
      do j=1,nlev
        C(l,j) = z(j,l)*DSQRT(d(l))
      enddo
      enddo
      C = MATMUL(z,C)
!     do l=1,nlev
!       write(*,'(10(2x,f5.3))')Cv(l,:)
!     enddo

      end subroutine VCORR_COEFF

! ======================================================================

      integer function XnmIndex(n,m)
! ----------------------------------------------------------------------
! This function finds the index of the vector of spherical harmonics given
! the indices n and m. The dimensions of Ynm are supposed to start at 1.
! The indices n and m are supposed to start at 0 and -n, respectively.
! Example:
! (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)
!   1     1,-1   1,0    1,1    2,-2   2,-1   2,0    2,1    2,2
!  Xnm    Xnm    Xnm    Xnm    Xnm    Xnm    Xnm    Xnm    Xnm
!   1      2      3      4      5      6      7      8      9   ....
! Authors: quentin@oma.be
! Date:     Jan 2011
! ----------------------------------------------------------------------

      implicit none
      integer, intent(in) :: n, m
      if (abs(m)>n) then
        XnmIndex = -1
      else
        XnmIndex = n*n + n + 1 + m
      endif

      end function XnmIndex

! ======================================================================

      integer function PnmIndex(n,m)
! ----------------------------------------------------------------------
! This function finds the index of the vector of associated Legendre
! functions given the indexes n and m. The dimension of Pnm are supposed 
! to start at 1. The indexe n is suppose to start at 0 and m is suppose to 
! start at 0.
! Example:
! (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)  (n,m)
!   1     1,0    1,1    2,0    2,1    2,2    3,0    3,1    3,2
!  Pnm    Pnm    Pnm    Pnm    Pnm    Pnm    Pnm    Pnm    Pnm
!   1      2      3      4      5      6      7      8      9   ....
! Authors: quentin@oma.be
! Date:     Jan 2011
! ----------------------------------------------------------------------

      implicit none
      integer, intent(in) :: n, m
      integer             :: nn, mm
      if (abs(m)>n .or. m<0) then
        PnmIndex = -1
      else
        PnmIndex = 1
        do nn=1,n
          do mm=0,nn-1
            PnmIndex = PnmIndex+1
          enddo
        enddo
        PnmIndex = PnmIndex+m
      endif

      end function PnmIndex

! ======================================================================

      subroutine S_DIRECT(Xs,x_sg)
! ----------------------------------------------------------------------
! This routine performs the transformation x = S(Xs) where x is
! defined on the spectral grid and Xs is defined in the spectral
! space. The routine PRECOMPUTE_BGLIB must be called prior this routine
! Input:
!   Xs            the (input) spectral coefficients
! Output:
!   x_sg          the (output) field on the spectral grid
!
! Authors: quentin@oma.be
! Date:     Oct 2010, Dec 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlon
      use BGLIB, only : nlat_sg, NX, Ns, Pnm, Eiml, lprecomp_bglib
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      complex*16, intent(in), dimension(NX) :: Xs
      real*8, intent(out), dimension(nlon,nlat_sg) :: x_sg
      complex*16 :: c_sg
      complex*16, dimension(nlat_sg,-Ns:Ns) :: Xil
      integer :: k, j, n, m, iP, iX, PnmIndex

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling S_DIRECT'
        print*,'Fatal error in S_DIRECT'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! Step 1: Legendre transform
! ----------------------------------------------------------------------
      Xil(:,:) = 0.d0
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( m, n, k, iP, iX )
      do m=-Ns,Ns,1
      do n=abs(m),Ns
#if (PRECOMP==1)
        iP = PnmIndex(n,abs(m))
#else
        iP = PlmIndex(n,abs(m))
#endif
        do k=1,nlat_sg
          if (m>=0) then
            Xil(k,m) = Xil(k,m) + Xs(iP) * Pnm(k,iP)
          else
            Xil(k,m) = Xil(k,m) + CONJG(Xs(iP)) * Pnm(k,iP)
          endif
        enddo
      enddo
      enddo
!$OMP END PARALLEL DO

! ----------------------------------------------------------------------
! Step 2: Fourier transform
! ----------------------------------------------------------------------
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( k, j, m, c_sg )
      do k=1,nlat_sg
      do j=1,nlon
        c_sg = (0.d0,0.d0)
        do m=-Ns,Ns
          c_sg = c_sg + Xil(k,m) * Eiml(m,j)
        enddo
        x_sg(j,k) = DBLE(c_sg)
      enddo
      enddo
!$OMP END PARALLEL DO

      end subroutine S_DIRECT

! ======================================================================

      subroutine S_INV(x_sg,Xs)
! ----------------------------------------------------------------------
! This routine performs the inverse transformation X = S^(-1)(x) where x is
! defined on the spectral grid and X is defined in the spectral
! space. The routine PRECOMPUTE_BGLIB must be called prior this routine
! Input:
!   x_sg          the (input) field on the spectral grid
! Output:
!   Xs            the (output) spectral coefficients
!
! Authors: quentin@oma.be
! Date:     Nov 2010, Dec 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlon
      use BGLIB, only : nlat_sg, NX, Ns, wg, Pnm, Eiml, lprecomp_bglib, lgg
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      real*8, intent(in), dimension(nlon,nlat_sg) :: x_sg
      complex*16, intent(out), dimension(NX) :: Xs
      integer :: k, j, n, m, im, iP, iX, PnmIndex
      complex*16, dimension(nlat_sg,-Ns:Ns) :: Xif

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling S_DIRECT'
        print*,'Fatal error in S_DIRECT'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
!     Check that physical grid is the Gaussian grid
! ----------------------------------------------------------------------
      if (.not. lgg) then
        print*,'Try to make inverse of spectal transform on latitudes which are not Gaussian'
        print*,'Fatal error in S_INV.'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! Step 1: Inverse Fourier transform.
! ----------------------------------------------------------------------
      Xif(:,:) = 0.d0
      im = -1
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( m, k, j )
      do m=-Ns,Ns,1
        do k=1,nlat_sg
        do j=1,nlon
          Xif(k,m) = Xif(k,m) + x_sg(j,k)  !* Eiml(-m,j)
!          print*,m,j,k,x_sg(j,k),Xif(k,m)
        enddo
        enddo
      enddo
!$OMP END PARALLEL DO
      do m=-Ns,Ns,1
        do k=1,nlat_sg
!          print*,m,k,Xif(k,m)
        enddo
      enddo
! ----------------------------------------------------------------------
! Step 2: Inverse Legendre transform.
! ----------------------------------------------------------------------
      Xs(:) = 0.d0
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( n, m, k, iP )
      do n=0,Ns
      do m=0,n
#if (PRECOMP==1)
        iP = PnmIndex(n,abs(m))
#else
        iP = PlmIndex(n,abs(m))
#endif
        do k=1,nlat_sg
          Xs(iP) = Xs(iP) + wg(k) * Xif(k,m) * Pnm(k,iP)
        enddo
      enddo
      enddo
!$OMP END PARALLEL DO

      end subroutine S_INV

! ======================================================================

      subroutine S_ADJ(x_sg,Xs)
! ----------------------------------------------------------------------
! This routine performs the adjoint operation of S_DIRECT X = S*(x) where x is
! defined on the spectral grid and X is defined in the spectral
! space. The routine PRECOMPUTE_BGLIB must be called prior this routine.
! Input:
!   x_sg          the (input) field on the spectral grid
! Output:
!   Xs            the (output) spectral coefficients
!
! Authors: quentin@oma.be
! Date:     Nov 2010, Dec 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlon
      use BGLIB, only : nlat_sg, NX, Ns, Pnm, Eiml, lprecomp_bglib
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      real*8, intent(in), dimension(nlon,nlat_sg) :: x_sg
      complex*16, intent(out), dimension(NX) :: Xs
      integer :: k, j, n, m, im, iP, iX, PnmIndex
      complex*16, dimension(nlat_sg,-Ns:Ns) :: Xif

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling S_ADJ'
        print*,'Fatal error in S_ADJ'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
! Step 1: Adjoint of the Fourier transform
! ----------------------------------------------------------------------
      Xif(:,:) = 0.d0
      im = -1
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( m, k, j )
      do m=-Ns,Ns,1
        do k=1,nlat_sg
        do j=1,nlon
          Xif(k,m) = Xif(k,m) + x_sg(j,k) * Eiml(-m,j)
        enddo
        enddo
      enddo
!$OMP END PARALLEL DO

! ----------------------------------------------------------------------
! Step 2: Adjoint of the Legendre transform
! ----------------------------------------------------------------------
      Xs(:) = 0.d0
!      write(*,'(a)')'  n  m  k     WW R(Xif) I(Xif)    Pnm  R(Xs)  I(Xs)'
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( n, m, k, iP )
      do n=0,Ns
      do m=0,n
#if (PRECOMP==1)
        iP = PnmIndex(n,abs(m))
#else
        iP = PlmIndex(n,abs(m))
#endif
        do k=1,nlat_sg
          Xs(iP) = Xs(iP) + Xif(k,m) * Pnm(k,iP)
        enddo
      enddo
      enddo
!$OMP END PARALLEL DO

      end subroutine S_ADJ

! ======================================================================

      subroutine SQRT_CORREL(ladjoint, v, Xs)
! ----------------------------------------------------------------------
! This routine performs the operation Xs = C^(1/2)Xs where Xs is a vector
! of spectral coeffients and C is the spatial correlation matrix defined 
! in the spectral space. The correlation matrix C^(1/2) is built in 
! PRECOMPUTE_BGLIB.
! Input/Output:
!   ladjoint      if .true., doing adjoint of C^1/2. If .false. doing forward.
!   v             index of the vmr vector
!   Xs            the input/output field in the spectral space
!
! Authors: quentin@oma.be
! Date:     Nov 2010
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlev
      use BGLIB, only : Ns, NX, Ch, Cv, lprecomp_bglib
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      logical, intent(in) :: ladjoint
      integer, intent(in) :: v
      complex*16, intent(inout), dimension(NX,nlev)  :: Xs
!      complex*16, dimension(NX,nlev) :: Xloc
      complex*16, dimension(nlev) :: Xnml
      integer :: n, m, l, j, iP, PnmIndex

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling SQRT_CORREL'
        print*,'Fatal error in SQRT_CORREL'
        call EXIT(1)
      endif

! ----------------------------------------------------------------------
!     Direct matrix operation
! ----------------------------------------------------------------------
      if (.not.ladjoint) then
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( l, j, n, m, iP, Xnml )
        do n=0,Ns
        do m=0,n
#if (PRECOMP==1)
          iP = PnmIndex(n,abs(m))
#else
          iP = PlmIndex(n,abs(m))
#endif
          do l=1,nlev
            Xs(iP,l) = Ch(n,l,v)*Xs(iP,l)
          enddo
          Xnml(:) = CMPLX(0.d0,0.d0,kind=8)
          do l=1,nlev
            do j=1,nlev
              Xnml(l) = Xnml(l) + Cv(l,j,v)*Xs(iP,j)
            enddo
          enddo
          Xs(iP,:) = Xnml(:)
        enddo
        enddo
!$OMP END PARALLEL DO
! ----------------------------------------------------------------------
!     Adjoint matrix operation
! ----------------------------------------------------------------------
      else
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( l, j, n, m, iP, Xnml )
        do n=0,Ns
        do m=0,n
#if (PRECOMP==1)
          iP = PnmIndex(n,abs(m))
#else
          iP = PlmIndex(n,abs(m))
#endif
          Xnml(:)  = CMPLX(0.d0,0.d0,kind=8)
          do l=1,nlev
            do j=1,nlev
              Xnml(l) = Xnml(l) + Cv(l,j,v)*Xs(iP,j)
            enddo
          enddo
          do l=1,nlev
            Xs(iP,l) = Ch(n,l,v)*Xnml(l)
          enddo
        enddo
        enddo
!$OMP END PARALLEL DO
      endif

      end subroutine SQRT_CORREL

! ======================================================================

      subroutine B_STDDEV(ilev, v, x_mg)
! ----------------------------------------------------------------------
! This routine performs the operation x_mg = E x_mg where x_mg is a model
! state vector on the model grid and E is the background error standard
! deviation.
! Input:
!   ilev          The model level used to get the background error
!                 standard deviation
!   v             index of the vmr vector
! Input/Output:
!   x_mg          the model state on the model grid at level 'ilev' and
!                 vmr index 'v'.
!
! Authors: quentin@oma.be
! Date:     Nov 2010
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlat, nlon, vmrfg
      use BGLIB, only : bg_std, bg_units, lprecomp_bglib
      implicit none
      integer, intent(in) :: ilev, v
      real*8, intent(inout), dimension(nlon,nlat) :: x_mg
      integer :: i

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling B_STDDEV'
        print*,'Fatal error in B_STDDEV'
        call EXIT(1)
      endif

      do i = 1, nlat
        if (bg_units=='vmr') then
          x_mg(:,i) = x_mg(:,i) *bg_std(ilev,v) !*vmrfg(:,k,ilev,1)
        elseif (bg_units=='rel') then
          x_mg(:,i) = x_mg(:,i) *bg_std(ilev,v)*vmrfg(:,i,ilev,v)
        endif
      enddo

      end subroutine B_STDDEV

! ======================================================================

      subroutine SG2MG(x_sg,x_mg)
! ----------------------------------------------------------------------
! This routine transforms the variable x_sg (on the spectral grid) to x_mg
! on the model grid.
! computed in PRECOMPUTE_BGLIB.
! Input:
!   x_sg          the model state on the spectral grid
! Output:
!   xmg           the model state on the model grid
!
! Authors: quentin@oma.be
! Date:     Nov 2010, DEC 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlat, nlon
      use BGLIB, only : nlat_sg, G, lprecomp_bglib, lgg, SpectralGrid
      implicit none
      real*8, intent(in), dimension(nlon,nlat_sg) :: x_sg
      real*8, intent(out), dimension(nlon,nlat) :: x_mg
      integer :: j, i, k

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling GAUSS2LL'
        print*,'Fatal error in GAUSS2LL'
        call EXIT(1)
      endif

      if (lgg) then
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( i, j)
        do j = 1, nlon
          do i = 1, nlat
            x_mg(j,i) = DOT_PRODUCT(G(i,:),x_sg(j,:))
          enddo
        enddo
!$OMP END PARALLEL DO
      else
        do k=1,nlat_sg
          x_mg(:,nlat-k+1) = x_sg(:,k)
        enddo
      endif

      end subroutine SG2MG

! ======================================================================

      subroutine AD_SG2MG(ad_mg,ad_sg)
! ----------------------------------------------------------------------
! This routine performs the adjoint of the transformation spectral grid
! to the model grid.
! Input:
!   ad_mg         the adjoint of J_obs on the model grid
! Output:
!   ad_sg         the adjoint of J_obs on the spectral grid
!
! Authors: quentin@oma.be
! Date:     Nov 2010
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlat, nlon
      use BGLIB, only : nlat_sg, G_TRANSP, lprecomp_bglib, lgg, SpectralGrid
      implicit none
      real*8, intent(in), dimension(nlon,nlat) :: ad_mg
      real*8, intent(out), dimension(nlon,nlat_sg) :: ad_sg
      integer :: j, k

! ----------------------------------------------------------------------
!     Check that precomp_bglib is done
! ----------------------------------------------------------------------
      if (.not.lprecomp_bglib) then
        print*,'BGLIB not precomputed while calling AD_GAUSS2LL'
        print*,'Fatal error in AD_GAUSS2LL'
        call EXIT(1)
      endif

      if (lgg) then
!$OMP PARALLEL DO SCHEDULE (dynamic) PRIVATE( j, k )
        do j = 1, nlon
          do k = 1, nlat_sg
            ad_sg(j,k) = DOT_PRODUCT(G_TRANSP(k,:),ad_mg(j,:))
          enddo
        enddo
!$OMP END PARALLEL DO
      else
        do k=1,nlat_sg
          ad_sg(:,k) = ad_mg(:,nlat-k+1)
        enddo
      endif

      end subroutine AD_SG2MG

! ======================================================================

      subroutine DOT_PROD_SPECTRAL(chi1,chi2,xdp)
! ----------------------------------------------------------------------
! This routine calculates the dot product of two real fields in spectral
! representation, i.e. two fields given only for m>=0 and where
!     Xn-m = (-1)**m CONJG(Xnm)     (i.e. a real field).
! Input:
!   Chi1          First spectral field (complex*16)
!   Chi2          Second spectral field (complex*16)
! Output:
!   xdp          The result of Chi1*CONJG(Chi2)
!
! Authors: quentin@oma.be
! Date:     Nov 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlev, nvmr
      use BGLIB, only : Ns, NX
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      integer, parameter :: NNN = (NS+1)*(Ns+1)
      complex*16, intent(in), dimension(NX,nlev,nvmr) :: chi1, chi2
      real*8, intent(out) :: xdp
      complex*16, dimension(NNN,nlev,nvmr) :: X1, X2
      complex*16 :: cdp
      integer :: l, n, m, v, iP, iX, XnmIndex, PnmIndex

! ----------------------------------------------------------------------
!     Make the dot product
! ----------------------------------------------------------------------
      cdp = 0.d0
      do l=1,nlev
      do v=1,nvmr
      do n=0,Ns
      do m=-n,n
#if (PRECOMP==1)
        iP = PnmIndex(n,abs(m))
#else
        iP = PlmIndex(n,abs(m))
#endif
        iX = XnmIndex(n,m)
        if (m>=0) then
          X1(iX,l,v) = chi1(iP,l,v)
          X2(iX,l,v) = chi2(iP,l,v)
        else
          X1(iX,l,v) = (-1)**m * DCONJG(chi1(iP,l,v))
          X2(iX,l,v) = (-1)**m * DCONJG(chi2(iP,l,v))
        endif
      enddo
      enddo
      enddo
      enddo
      xdp = DOT_PRODUCT( RESHAPE( X1, (/NNN*nlev*nvmr/) ) , RESHAPE( X2, (/NNN*nlev*nvmr/) ) )

      end subroutine DOT_PROD_SPECTRAL

! ======================================================================

      subroutine RESHAPE_M1QN3(perm, xg, Xs)
! ----------------------------------------------------------------------
! This routine reshapes the optimized M1QN3 variable xg to Xs and the
! reverse reshaping.
! Input:
!   perm          'vec2mat' or 'mat2vec'. In the first case, reshape
!                 xg => Xs. In the second case, reshape Xs => xg.
! Input/Output:
!   xg            If perm='vec2mat', xg is the input in the shape (nmax).
!                 If perm='mat2vec', xg is the output in the shape (nmax).
!   Xs            If perm='vec2mat', Xs is the output in the shape (NX,nlev,nvmr)
!                 If perm='mat2vec', xg is the intput in the shape (NX,nlev,nvmr)
!
!
! Authors: quentin@oma.be
! Date:     Nov 2010, Dec 2011
! ----------------------------------------------------------------------
      use VAR_DEF, only : nlev, nvmr
      use BGLIB, only : nmax, NX, Ns
#if (PRECOMP != 1)
      use SHTOOLS, only : PlmIndex
#endif
      implicit none
      character(len=*), intent(in) :: perm
      real*8, intent(inout), dimension(nmax) :: xg
      complex*16, intent(inout), dimension(NX,nlev,nvmr) :: Xs
      integer :: l, v, n, m, ix, ig, iP, PnmIndex
      complex*16 :: Xnm

! ----------------------------------------------------------------------
! ----------------------------------------------------------------------
      if (perm=='vec2mat') then
        ix = 0
        do l=1,nlev
        do v=1,nvmr
        do n=0,Ns
        do m=0,n
          ix = ix+1
          if (m>=0) then
#if (PRECOMP==1)
            iP = PnmIndex(n,abs(m))
#else
            iP = PlmIndex(n,abs(m))
#endif
            Xs(iP,l,v) = CMPLX(xg(ix),xg(ix+nmax/2),kind=8)
          endif
        enddo
        enddo
        enddo
        enddo
      elseif (perm=='mat2vec') then
        ig = 0
        do l=1,nlev
        do v=1,nvmr
        do n=0,Ns
        do m=0,n
          ig = ig+1
#if (PRECOMP==1)
          iP = PnmIndex(n,abs(m))
#else
          iP = PlmIndex(n,abs(m))
#endif
          if (m>=0) then
            Xnm = Xs(iP,l,v)
          else
            Xnm = (-1)**m * DCONJG(Xs(iP,l,v))
          endif
          xg(ig) = DBLE(Xnm)
          xg(ig+nmax/2) = DIMAG(Xnm)
        enddo
        enddo
        enddo
        enddo
      else
        print*,'Unkown request operation in RESHAPE_M1QN3'
        print*,'perm=>'//perm//'<'
        call EXIT(1)
      endif

      end subroutine RESHAPE_M1QN3

