#! /bin/tcsh -ef
################################################################################
# set computer specific shell variables
# set gnuplot path
set gnuplot="/usr/local/bin/gnuplot"
set gnuplot="/opt/local/bin/gnuplot"
# set ps2pdf path
set ps2pdf="/opt/local/bin/ps2pdf"
set ps2pdf="/usr/local/bin/ps2pdf"
# set open command to view pdf file
set open="ls -l"
set open="/usr/bin/open"
# set compiler with options
set f90="ifort -cpp -g -O0 -traceback -check all -debug extended"
################################################################################
# choose parameterization "Eq17b" or "Kappa"
set para="Kappa"
set para="Eq17b"
################################################################################
# choose compound "NH42SO4" or "NaCl"
set speci="NH42SO4"
set speci="NaCl"
################################################################################
# set dry diameter [micro m]
set d="1"
################################################################################
set prog="test_${para}_${speci}_${d}um"
################################################################################
# compound specification ...
if (${speci} == "NaCl") then
  set ns="1.000e-6"        # solute "concentration" [mol] - w.r.t. per m3(air)
  set Ds="2170.000"        # density [kg/m3]
  set Ms="0.058440"        # molar mass [kg/mol]
  set Ws="0.264700"        # mass fraction solubility [-]
  set RHDrefTc="25.00"     # reference DRH T-coefficient [-]
  set RHDrefT0="0.7528"    # reference deliquescence reative humidity (DRH) at T0 [-]
  set kappa="1.28"         # kappa=1.28  - value [-] according to Rose et al., 2008
  set vi="1.737506"        # vi=1.737506 - value [-] according to Metzger et al., 2011
endif
if (${speci} == "NH42SO4") then
  set ns="1.000e-6"        # solute "concentration" [mol] - w.r.t. per m3(air)
  set Ds="1770.000"        # density [kg/m3]
  set Ms="0.132140"        # molar mass [kg/mol]
  set Ws="0.433100"        # mass fraction solubility [-]
  set RHDrefTc="80.00"     # reference DRH T-coefficient [-]
  set RHDrefT0="0.7997"    # reference deliquescence reative humidity (DRH) at T0 [-]
  set kappa="0.61"         # kappa=0.61  - value [-] according to Rose et al., 2008
  set vi="1.661410"        # vi=1.661410 - value [-] according to Metzger et al., 2011
endif
################################################################################
echo "${speci} - dry diameter = ${d} at T0 = 298.15 K with RHDrefT0 = ${RHDrefT0}" using ${para} method
################################################################################
cat > ./${prog}.f90 <<EOF
PROGRAM test_Eq17b_Kappa
!
! *** automatically generated with test_Eq17b_Kappa.j ***
!
! AUTHOR
! ------
! Swen Metzger <swen.metzger@mpic.de>
!
!      Department of Atmospheric Chemistry,
!      Max Planck Institute for Chemistry, Mainz, Germany.
!      Copyright 2011+. All rights reserved.
!
!      References
!      ---------
!      [1] CRC Hanbook of Chemistry and Physics, 85th Edition, 2004-2005.
!
!      [2] Seinfeld and Pandis, 1998.
!          Atmospheric chemistry and physics, J. Wiley and Sons, Inc., New York.
!
!      [3] Rose et al., 2008.
!          Calibration and measurement uncertainties of a continuous-flow 
!          cloud condensation nuclei counter (DMT-CCNC): activation of ammonium 
!          sulfate and sodium chloride aerosol particles in theory and experiment,
!          Atmos. Chem. Phys., 8, 1153–1179, www.atmos-chem-phys.net/8/1153/2008/
!
!      [4] Metzger et al., 2011;
!          Aerosol hygroscopic growth parameterizations, submitted to ACPD, 2011.
!------------------------------------------------------------------------------------
IMPLICIT NONE
!
INTEGER,PARAMETER  :: dp = SELECTED_REAL_KIND(12,307)
!--------------------------------------------------------------------------------
INTEGER              :: i,j,k                       ! loop indices
INTEGER, PARAMETER   :: imax = 686                  ! maximum loop count (100 + 586 of Rose et. al)
INTEGER, PARAMETER   :: IUNIT = 1                   ! I/O Unit number - read
INTEGER, PARAMETER   :: OUNIT = 2                   ! I/O Unit number - write
!--------------------------------------------------------------------------------
REAL(dp),PARAMETER   :: Dd=${d}_dp*1.e-6_dp         ! dry diameter [m]
REAL(dp),PARAMETER   :: RHDrefT0=${RHDrefT0}_dp     ! reference RHD [-]
REAL(dp),PARAMETER   :: RHDrefTc=${RHDrefTc}_dp     ! reference RHD T-coefficient [-]
REAL(dp),PARAMETER   :: kappa=${kappa}_dp           ! Kappa value [-], Eq. (30) of [3]
REAL(dp),PARAMETER   :: vi=${vi}_dp                 ! vi value [-], Eq. (15) of [4]
!--------------------------------------------------------------------------------
REAL(dp),PARAMETER   :: Phi=3.141592653589793_dp    ! Phi [-]
REAL(dp),PARAMETER   :: T0=298.15_dp                ! reference temperature, T0 [K]
REAL(dp),PARAMETER   :: R=8.314_dp                  ! universal gas constant [J/K/mol]
REAL(dp),PARAMETER   :: sigma=0.0761_dp             ! surface tension of H2O at 273 K [N/m]
REAL(dp),PARAMETER   :: T=298.15_dp                 ! default 298.15 temperature, T [K]
!--------------------------------------------------------------------------------
! compound specification; thermodynamic data of [1]
REAL(dp),PARAMETER   :: ns=${ns}_dp                 ! solute "concentration" [mol] - w.r.t. per m3(air)
REAL(dp),PARAMETER   :: Ds=${Ds}_dp                 ! solute density [kg/m3]
REAL(dp),PARAMETER   :: Dw=997.1_dp                 ! water  density [kg/m3]
REAL(dp),PARAMETER   :: Ms=${Ms}_dp                 ! molar mass solute [kg/mol]
REAL(dp),PARAMETER   :: Mw=0.018015_dp              ! molar mass H2O  [kg/mol]
REAL(dp),PARAMETER   :: Ws=${Ws}_dp                 ! mass fraction solubility [-]; Ws=ms/(ms+mw); ms=mw/(1/Ws-1)
REAL(dp),PARAMETER   :: mwat=1._dp                  ! initial mass of H20 [1 kg]
REAL(dp),PARAMETER   :: msat=mwat/(1._dp/Ws-1._dp)  ! saturation mass [kg] that completely dissolves at equilbrium RHD
REAL(dp),PARAMETER   :: nsat=msat/Ms                ! number of moles of solute in a saturated solution at RHD
REAL(dp)             :: RHD,RHDref                  ! calculated and reference RHD at T and T0 [-]
REAL(dp)             :: sm,sm0                      ! solute molality, saturation molality at RHD[mol/kg(H2O)], Eq.(11)
REAL(dp)             :: aw                          ! water activity [-]
REAL(dp)             :: A                           ! A term [-], Eq.(15) of [4]
REAL(dp)             :: B                           ! B term [-], Eq.(15) of [4]
REAL(dp)             :: WH2O                        ! calculated aerosol associated water mass [ug(H2O)/m3(air)]
REAL(dp)             :: GF                          ! diameter growth factor [0-1], Eq.(2) of [4]
REAL(dp)             :: Ke                          ! Kelvin term [-], Eq. (1) of [4]
REAL(dp)             :: Dpw                         ! wet diameter [m]
REAL(dp)             :: Sc_ref                      ! E-AIM   critical supersaturation [-]
REAL(dp)             :: Sc_cal                      ! ${para} critical supersaturation [-]
REAL(dp),DIMENSION(imax):: RH                       ! calculated RH from ${para}
REAL(dp),DIMENSION(imax):: &                        ! E-AIM reference data:
                           AWref, &                 ! water activity  (input)
                           SMref, &                 ! solute molality (input)
                           WH2Oref,&                ! derived water mass
                           GFref, &                 ! derived growth factor
                           Dpwref,&                 ! derived wet diameter
                           Keref, &                 ! derived Kelvin-term
                           RHref                    ! derived RH
!------------------------------------------------------------------------------------
WRITE(*,'(A)') '(I) reading / calculating E-AIM reference data ...'
!------------------------------------------------------------------------------------
OPEN(unit=IUNIT,file= './input/${speci}_E-AIM_inp.txt',       status='old',form='formatted')
OPEN(unit=OUNIT,file='./output/${speci}_E-AIM_out_${d}um.txt',status='new',form='formatted')
! calculate T-dependency of RHDref
RHDref = MAX(0._dp,MIN(1._dp,RHDrefT0*EXP(RHDrefTc*(1._dp/T-1._dp/T0))))
! initialize
AWref  (:) = 0._dp
WH2Oref(:) = 0._dp
GFref  (:) = 1._dp
DO i=1,imax
   ! read water activity (aw) and solute molality (sm)
   READ(IUNIT,'(2E16.9)') AWref(i),SMref(i)
   ! calculate water mass from solute molality - Eq. (11) of [4]
   IF(SMref(i) > 0._dp) &
   WH2Oref (i)   = ns / SMref(i)
   ! calculate mass equivalent (diameter) growth factor - Eq. (2) of [4]
   IF(WH2Oref(i) > 0._dp) &
   GFref   (i)   = (Ds*WH2Oref(i)/(ns*Ms*Dw)+1)**(1._dp/3._dp)
   ! calculate Ke from GF using Eq. (2) of [4]
   Keref   (i)   = exp(4._dp*Mw*sigma/(R*T*Dw*GFref(i)*Dd))
   ! calculate RH from Eq. (1) of [4]
   RHref   (i)   = AWref(i)*Keref(i)
   ! set growth factor to unity if RH is below the compound's RHD
   IF(RHref(i)   < RHDref*Keref(i)) &
   GFref   (i)   = 1._dp
   ! calculate wet radius from growth factor - Eq. (2) of [4]
   Dpwref  (i)   = GFref(i)*Dd
   WRITE(OUNIT,'(I6,10E18.9)') i,AWref (i), RHref(i)*100._dp,WH2Oref(i)*1.e9_dp, Keref(i),&
                                Dpwref(i)*1.e6_dp,(RHref(i)-1._dp)*100._dp,GFref(i),Dd*1.e6_dp,SMref(i),0._dp
END DO
CLOSE(IUNIT)
CLOSE(OUNIT)
WRITE(*,'(A)') 'finished ...'
!------------------------------------------------------------------------------------
IF     ("${para}" == "Eq17b") THEN
   WRITE(*,'(A)') '(II) solving Eq.(17b) of [4] as a function of wet diameter using ${para} method ...'
ELSE IF("${para}" == "Kappa") THEN
   WRITE(*,'(A)') '(II) solving Eq.(30)  of [3] as a function of wet diameter using ${para} method ...'
ELSE
   WRITE(*,'(A)') 'choose parameterization "Eq17b" or "Kappa"'
   STOP
END IF
!------------------------------------------------------------------------------------
OPEN(unit=OUNIT,file='./output/${speci}_${para}_out_${d}um.txt',status='new',form='formatted')
! initialize
! growth factor
GF = 1._dp
! Kelvin term
Ke = 1._dp
! A term
A  = 1._dp
! B term
B  = 0._dp
! aerosol water mass
WH2O = 0._dp
! wet diameter = dry diameter
Dpw  = Dd
! saturation molality at RHD
sm0= nsat
! (re-)calculate Ke from GF using Eq. (2) of [4]
Ke = exp(4._dp*Mw*sigma/(R*T*Dw*GF*Dd))
!-----------------------------------------------------------------
IF     ("${para}" == "Eq17b") THEN
   ! calculate A-term - Eq. (18) of [4]
   A = exp(-Mw*vi*sm0**vi)*(1._dp+Mw*vi*sm0)
   ! calculate B-term - Eq. (19) of [4]
   B = 1._dp/(1._dp+1._dp/(Ms*vi*sm0))*(vi*sm0)**(-1._dp/vi)
   ! calculate RHD with vi method - Eq. (16b) of [4]
   RHD = Ke/(A+Mw*vi*(sm0+B)**vi)
ELSE IF("${para}" == "Kappa") THEN
   ! calculate RHD with Kappa method - Eq. (30) of [3]
   RHD = Ke*(Dpw**3._dp-Dd**3._dp)/(Dpw**3._dp-Dd**3._dp*(1._dp-kappa))
END IF
WRITE(*,'(A,4F12.6)')' ${speci} - RHDref,RHD,vi,kappa ',RHDref,RHD,vi,kappa
DO i = 1,imax
   ! assign wet diameter [m] (derived from E-AIM solute molality)
   Dpw = Dpwref(i)
   ! (re-)calculate mass equivalent (diameter) growth factor - Eq. (2) of [4]
   GF = Dpw / Dd
   ! (re-)calculate Ke from GF using Eq. (2) of [4]
   Ke = exp(4._dp*Mw*sigma/(R*T*Dw*GF*Dd))
   ! initialize solute molality from saturation molality (solubility) - Eq. (11) of [4]
   sm  = sm0
   ! (re-)calculate solute molality from growth factor - Eq. (2) of [4]
   IF(GF > 1._dp) &
   sm = 1._dp/((GF**3._dp-1._dp)*Ms*Dw/Ds)
   IF     ("${para}" == "Eq17b") THEN
      ! calculate A-term - Eq. (18) of [4]
      A = exp(-Mw*vi*sm**vi)*(1._dp+Mw*vi*sm)
      ! calculate B-term - Eq. (19) of [4]
      B = 1._dp/(1._dp+1._dp/(Ms*vi*sm))*(vi*sm)**(-1._dp/vi)
      ! calculate RH using vi method - Eq. (16a) of [4]
      RH(i) = Ke/(A+Mw*vi*(sm+B)**vi)
   ELSE IF("${para}" == "Kappa") THEN
      ! calculate RH using kappa method - Eq. (30) of [3]
      RH(i) = Ke*(Dpw**3._dp-Dd**3._dp)/(Dpw**3._dp-Dd**3._dp*(1._dp-kappa))
   END IF
   ! calculate water activity from RH/Ke - Eq. (1) of [4]
   aw = RH(i) / Ke
   ! calculate aerosol water mass [kg/m3(air)] - Eq. (2c) of [4]
   WH2O = ns / sm
   WRITE(OUNIT,'(I6,11E18.9)') i,aw,  RH(i)*100._dp,WH2O*1.e9_dp,Ke,&
                               Dpw*1.e6_dp,(RH(i)-1._dp)*100._dp,GF,Dd*1.e6_dp,sm,vi,kappa
END DO
CLOSE(OUNIT)
! pick critical supersaturation according to [2]
Sc_ref = 0._dp
DO i=1,imax
   IF(RHref(i) > Sc_ref) THEN
      Sc_ref   = RHref(i)
   END IF
END DO
Sc_ref = Sc_ref*100._dp-100._dp
Sc_cal = 0._dp
DO i=1,imax
   IF(RH(i) > Sc_cal) THEN
      Sc_cal   = RH(i)
   END IF
END DO
Sc_cal = Sc_cal*100._dp-100._dp
! open output file
OPEN(unit=OUNIT,file='./output/${speci}_${para}_out_${d}um_Sc.txt',status='new',form='formatted')
WRITE(OUNIT,'(3E15.6)') Sc_ref,Sc_cal,Dd*1.e6_dp
CLOSE(OUNIT)
WRITE(*,'(A)') 'finished ...'
STOP
!_________________________________________________________________________________________________________________________________
END PROGRAM test_Eq17b_Kappa
EOF
################################################################################
# create output directory (if not existing)
mkdir -p ./output
# remove calculated E-AIM  output (if any)
rm -f ./output/${speci}_E-AIM_out_${d}um.txt
# remove calculated${para} output (if any)
rm -f ./output/${speci}_${para}_out_${d}um_Sc.txt
rm -f ./output/${speci}_${para}_out_${d}um.txt
# remove executable
rm -f ./a.out
# compile program
${f90} ./${prog}.f90
# list input
ls -l ./input/${speci}_E-AIM_inp.txt
# excecute program
./a.out
# list calculated output
ls -l ./output/${speci}_${para}_out_${d}um.txt
ls -l ./output/${speci}_E-AIM_out_${d}um.txt
ls -l ./output/${speci}_${para}_out_${d}um_Sc.txt
# cat Sc  output
cat   ./output/${speci}_${para}_out_${d}um_Sc.txt
#exit
################################################################################
# assign gnuplot title info
if (${para} == "Kappa") then
  set parameter="Kappa"
  set parameter_value=${kappa}
endif
if (${para} == "Eq17b") then
  set parameter="vi"
  set parameter_value=${vi}
endif
################################################################################
# generate gnuplot file
cat > ./${prog}.plot << EOF
set term postscript color font "arial,14"
set termoption enhanced
set output '${prog}.ps'
set grid
set tics
set key at graph 0.225,0.975
set style line  1 lt  1 lw 1.0 pt  1 ps 1.00
set style line  2 lt  2 lw 1.0 pt  4 ps 1.00
set style line  3 lt  3 lw 1.0 pt  6 ps 1.00
set style line  4 lt  4 lw 1.0 pt  5 ps 1.00
set title   "${speci} with dry diameter d=${d}[{/Symbol m}m] at T=298.15 [K] - ${parameter} = ${parameter_value}"
set ylabel   "Growth factor [-]"
set xlabel   "RH [%]"
set xrange  [70.0:97]
set yrange  [1:3.5]
set nologscale
plot "./output/${speci}_E-AIM_out_${d}um.txt"    using 3: 8 ti "E-AIM"   w points ls 3, \
     "./output/${speci}_${para}_out_${d}um.txt"  using 3: 8 ti "${para}" w points ls 1
set ylabel   "Growth factor [-]"
set xlabel   "RH [%]"
set xrange  [97.0:99.9]
set yrange  [1:7.5]
set nologscale
plot "./output/${speci}_E-AIM_out_${d}um.txt"    using 3: 8 ti "E-AIM"   w points ls 3, \
     "./output/${speci}_${para}_out_${d}um.txt"  using 3: 8 ti "${para}" w points ls 1
set ylabel   "Supersaturation [%]"
set xlabel   "Wet diameter [{/Symbol m}m]"
set xrange  [0.001:1000]
set yrange  [-1.0:20]
set yrange  [-1.0:1]
set logscale x
plot "./output/${speci}_E-AIM_out_${d}um.txt"    using 6: 7 ti "E-AIM"   w points ls 3, \
     "./output/${speci}_${para}_out_${d}um.txt"  using 6: 7 ti "${para}" w points ls 1
EOF
################################################################################
#exit
################################################################################
# make gnuplot
${gnuplot} ./${prog}.plot
################################################################################
# make pdf
${ps2pdf} ./${prog}.ps
################################################################################
# open pdf
${open} ./${prog}.pdf
################################################################################
# clean up
rm -f ./a.out ./${prog}.ps ./${prog}.plot #./${prog}.f90
################################################################################
exit

