#! /bin/tcsh -ef
pwd
set SYSTEM = `uname`
if ($SYSTEM   == "Darwin") then
 set f90=ifort
 set open=open
else if ($SYSTEM   == "Linux") then
 set f90=lf95
 set open=gv
else
 set f90=f95
 set open=gs
endif
#_______________________________________________________________________________
cat > ./testEQSAM3.f90 <<EOF
PROGRAM testEQSAM3
!
! DESCRIPTION
! -----------
! Test the EQSAM3 box model and concept w.r.t. NaCl
!
! AUTHOR
! ------
! Swen Metzger <swen.metzger@mpic.de>, 16. June 2010
!      Department of Atmospheric Chemistry,
!      Max Planck Institute for Chemistry, Mainz, Germany.
!      Copyright 2010+. All rights reserved.
!
! For more details see:
!
! (1) http://www.mpch-mainz.mpg.de/~metzger/emac/eqsam3.html
! (2) http://www.atmos-chem-phys.net/7/3163/2007/acp-7-3163-2007.html
! (3) http://www.atmos-chem-phys-discuss.net/10/8165/2010/acpd-10-8165-2010.html
!_______________________________________________________________________________
IMPLICIT NONE
!
INTEGER,PARAMETER  :: dp = SELECTED_REAL_KIND(12,307)
!
INTEGER, PARAMETER   :: IO = 1 ! I/O Unit number
INTEGER              :: i      ! loop index
REAL(dp)             :: sw,RH,RHD,WH2O,vw
! check water uptake of NaCl]; these values are explained in (2) and (3)
REAL(dp),PARAMETER   :: ns=1.e-6_dp                 ! NaCl "concentration"  [mol] - concentration w.r.t. per m3(air)
REAL(dp),PARAMETER   :: Ms=58.440_dp                ! molar mass NaCl [g/mol]
REAL(dp),PARAMETER   :: Mw=18.015_dp                ! molar mass H2O  [g/mol]
REAL(dp),PARAMETER   :: Ws=0.2647_dp                ! mass fraction solubility of NaCl [-]; Ws=ms/(ms+mw); ms=mw/(1/Ws-1)
REAL(dp),PARAMETER   :: mmw=1000._dp                ! initial mass of H20  [g]
REAL(dp),PARAMETER   :: mms=mmw/(1._dp/Ws-1._dp)    ! maximum mass of NaCl [g] that can be dissolved  in 1 kg H2O at saturation
REAL(dp),PARAMETER   :: nsat=mms/Ms                 ! number of moles of NaCl in a saturated solution of 1 kg water [mol]
REAL(dp),PARAMETER   :: nwat=mmw/Mw                 ! inverse mass fraction of 1 mole of H2O in 1 kg water = 55.51[kg/kg]
REAL(dp),PARAMETER   :: ve=2._dp                    ! effective stoichiometric coefficient of solute NaCl [-]
REAL(dp),PARAMETER   :: RHmax=95._dp                ! maximum RH [%]
!
OPEN(unit=IO,file='./testEQSAM3.txt', status='unknown',form='formatted')
!
vw=log10(2._dp/ve*Ws)+2._dp                ! effective stoichiometric coefficient of water [-]]; Eq. A2 of (3)
RHD=1._dp/(ve/vw*nsat**(ve/vw)/nwat+1._dp) ! deliquescence humidity of NaCl [-]]; Eq. A10 of (3)
!
WRITE(IO,'(6(A,F13.7))') '# - ns', ns, ' - Ms', Ms, ' - mms', mms, ' - nsat', nsat, ' - ve', ve, ' - RHD  ', RHD
WRITE(IO,'(6(A,F13.7))') '# - Ws', Ws, ' - Mw', Mw, ' - mmw', mmw, ' - nwat', nwat, ' - vw', vw, ' - RHmax', RHmax
!
DO i=1,int(RHmax)
   ! fractional relative humidity
   RH = float(i)/100._dp
   ! solute activity [kg/kg(H2O)]; Eq. A8 of (3)
   ! sw = (vw/ve*nwat*(1._dp/RH-1._dp))**(vw/ve)
   ! update / new interpretation of the solute activity Eq. A8 of (3);
   ! dimensions now in [kg/kg], since nwat and Ws are both mass fractions; 
   ! the differcence yield the remaining free water - required instead of total water
   ! initial water mass mmw = 1kg
   ! solute bound water   in the solution: mbw=mmw*(1-Ws)
   ! remaining free water in the solution: mfw=mmw-mbw
!  sw = (vw/ve*nwat*(1._dp/RH-1._dp)-2._dp/ve*Ws)**(vw/ve)
   sw = (vw/ve*nwat*(1._dp/RH-1._dp)-ve/vw*Ws)**(vw/ve)
   ! aerosol associated water mass [ug(H2O)]]; Eq. A9 of (3)
   IF(RH >= RHD) & 
   WH2O = ns / sw * 1.e9_dp
   WRITE(IO,'(I3,3F15.6)') i, RH*100, sw, WH2O
END DO
!
CLOSE(IO)
!
STOP
!
END PROGRAM testEQSAM3
EOF
#_______________________________________________________________________________
$f90   ./testEQSAM3.f90 -o testEQSAM3.exe
       ./testEQSAM3.exe 
cat    ./testEQSAM3.txt
#_______________________________________________________________________________
cat > ./testEQSAM3.plot << EOF
set term postscript color
set output 'testEQSAM3_NaCl.ps'
set grid
set title   "1 [umol/m3(air)] NaCl (single solution)"
set ylabel   "Aerosol Water [ug/m3(air)]"
plot "testEQSAM3.txt" using   2: 4 ti "EQSAM3", \
     "E-AIM_NaCl.txt" using  18:19 ti "E-AIM" 
EOF
#_______________________________________________________________________________
gnuplot ./testEQSAM3.plot
#gs      ./testEQSAM3_NaCl.ps
rm testEQSAM3.exe testEQSAM3.f90 testEQSAM3.plot
#exit
ps2pdf  ./testEQSAM3_NaCl.ps
rm      ./testEQSAM3_NaCl.ps
$open   ./testEQSAM3_NaCl.pdf
#_______________________________________________________________________________
exit
#_______________________________________________________________________________

