# Numerical implementation of the ice crystal number and contrail depth parametrization 
# The approximations are based on the parametrizations presented in Unterstrasser, 2016, ACP, "Properties of young contrails..." (U2016)
# The parametrization was updated in Lottermoser & Unterstrasser (2024, in rev.) (LU2025)
# Note: If e_water and/or Gamma are not available, simple estimates based on b are used.
# written by Annemarie Lottermoser and Simon Unterstrasser, DLR Oberpfaffenhofen
# If you want to use the program, please contact: simon.unterstrasser@dlr.de

import numpy as np

def esat(T):
    # input: ambient temperature (in K)
    # output: water vapour saturation pressure over ice in Pa
    return np.exp(9.550426-5723.265/T + 3.53068*np.log(T) - 0.00728332*T)

def zbuffer(iversion,T, rho_emit, rhi):
    # inputs: 
    # T: ambient temperature (in K), 
    # rho_emit: initial water vapor concentration surplus (in kg/m^3), 
    # rhi: relative humidity over ice (in %)
    # outputs: zatm in m (if rho_emit is 0 kg/m^3) or zemit in m (if rhi=100%)

    # Ensure T is always an array
    T = np.atleast_1d(T)

    #adiabatic index
    if iversion==1:
        kappa=1
    else:
        kappa=3.5
        
    rho_emit = rho_emit / (T**(kappa-1))

    si = (rhi - 100.0) / 100.0 if rhi >= 100.0 else (rhi / 100.0 if rhi >= 1.0 else rhi)
    
    R_v = 461.5

    for i,t in enumerate(T):

        rhov_links = (1 + si) * esat(t) / (R_v * t**kappa)
        rho_tot = rhov_links + rho_emit[i]
        dT = [0.0, 50.0]
        n_iterations = 100

        for _ in range(n_iterations):
            T = t + (dT[0] + dT[1]) / 2
            x = esat(T) / (R_v * T**kappa)
            if x >= rho_tot:
                dT[1] = (dT[0] + dT[1]) / 2
            else:
                dT[0] = (dT[0] + dT[1]) / 2

        result = (dT[0] + dT[1]) / 2 / 9.8 * 1000.0

    return result

def zdesc(Gamma, Nbv):
    # inputs: 
    # Gamma: initial circulation (in m^2/s),
    # Nbv: ambient stratification (in 1/s)
    # output: zdesc in m
    return np.sqrt(8*Gamma/(np.pi*Nbv))

def radius_eff(b0):
    # input: 
    # b0: intial separation of vortices (in m)
    # output: plume radius in m at intermediate stage (Eq. A6 in U2016)
    alpha_plume_area = 0.4
    gamma_plume_area = 1.5
    return alpha_plume_area * b0 + gamma_plume_area

    
def zdelta_2025(zdesc, zatm, zemit, N0, Ap, alpha_fit_desc, alpha_fit_atm, alpha_fit_emit, gamma_fit):
    # inputs: 
    # zdesc, zatm, zemit: length scales (in m), 
    # N0: initial number of ice crystals (in 1/m), 
    # Ap: plume cross-sectional area (in m^2),
    # alpha_fit_desc, alpha_fit_atm, alpha_fit_emit, gamma_fit: fit coefficients
    # ouput: zdelta parameter (in m) (Eq. 9 in LU2025)
    b0 = np.pi/4*60.3 #reference wingspan
    Ap_ref = 2*np.pi*radius_eff(b0)**2 #reference plume area
    n_ref = 3.38e12 / Ap_ref #reference ice crystal number concentration
    n = N0/Ap
    nstar = n/n_ref
    prefactor = 1/nstar
    zdelta = -alpha_fit_desc * zdesc + prefactor**gamma_fit * (alpha_fit_atm * zatm + alpha_fit_emit * zemit)
    return zdelta
    
def zdelta_2016(zdesc, zatm, zemit, EI_iceno, alpha_fit_desc, alpha_fit_atm, alpha_fit_emit, gamma_fit):
    # inputs: 
    # zdesc, zatm, zemit: length scales (in m),
    # EI_iceno: ice crystal emission index (in 1/kg)
    # alpha_fit_desc, alpha_fit_atm, alpha_fit_emit, gamma_fit: fit coefficients
    # ouput: zdelta parameter (in m) (Eq. 3 in U2016)
    EI_iceno_ref = 2.8e14
    prefactor = EI_iceno_ref/EI_iceno
    zdelta = -alpha_fit_desc * zdesc + prefactor**gamma_fit * (alpha_fit_atm * zatm + alpha_fit_emit * zemit)
    return zdelta

def arctangent_function(x, A, C, D):
    arctangens = A/np.pi * np.arctan(x + C) + D
    return np.maximum(0, arctangens)

def contrail_param(iversion,T,rhi,EI_iceno,Nbv,b,e_water,Gamma,EI_H2O):
    # inputs: iversion (1: 2016 param, 2: 2025 param, 3: 2025 param with fit formulae for zatm and zemit), 
    # T: ambient temperature (in K), 
    # rhi: ambient relative humidity over ice (in %),
    # EI_iceno: ice crystal "emission" index (in 1/kg),
    # Nbv: ambient stratification (in 1/s), 
    # b: aircraft wingspan (in m), 
    # e_water: initial amount of emitted water vapor (in g/m) (if unspecified, set it to 0),
    # Gamma: initial circulation (in m^2/s) (if unspecified, set it to 0), 
    # EI_H2O: water vapour emission index (in kg/kg) (if unspecified, set it to 0)
    # outputs: parameterized survival fraction of ice crystals (in 1), parameterized contrail height after vortex phase (in m), 
    # zdelta parameter (in m), descent length scale (in m), atmospheric length scale (in m), emission length scale (in m)
    if iversion==1:
        # fit parameters according to param 2016
        beta0,beta1,alpha0 = 0.45,1.19,-1.35
        alpha_desc,alpha_atm,alpha_emit,gamma = 0.6,1.70,1.15,0.18
        prefactor_Ap = 4
    else:
        # fit parameters according to param 2025
        beta0,beta1,alpha0 = 0.42, 1.31, -1.0
        alpha_desc,alpha_atm,alpha_emit,gamma = 0.49, 1.27, 0.42, 0.16
        prefactor_Ap = 2

    x_s,eta1,eta2 = 0.2,6.0,0.15
    b0 = np.pi/4*b
    r_eff = radius_eff(b0)
    a_eff = prefactor_Ap*np.pi*r_eff**2
    if e_water==0:
        e_water = 1e3*3.1250e-06 *b**2 # formula A9 in U2016, output in g/m
    if Gamma==0:
        Gamma = 10*b-70 # formula A5 in U2016, output in m^2/s
    if EI_H2O==0:
        EI_H2O = 1.26
    e_water_kgm = e_water*1e-3
    rho_emit = e_water_kgm / a_eff
    zdesc_comp = zdesc(Gamma, Nbv)
    if iversion==1 or iversion==2:
        zatm_comp = zbuffer(iversion, T, 0, rhi)
        zemit_comp = zbuffer(iversion, T, rho_emit, 100.0)
    else:
        s_i = (rhi - 100.0) / 100.0 if rhi >= 100.0 else (rhi / 100.0 if rhi >= 1.0 else rhi)
        zatm_comp = zatm_fit(s_i,T)
        zemit_comp = zemit_fit(rho_emit,T)

    N_form = e_water_kgm/EI_H2O*EI_iceno

    if iversion==1:
        # 2016 version
        zdelta_comp = zdelta_2016(zdesc_comp,zatm_comp,zemit_comp,EI_iceno,alpha_desc,alpha_atm,alpha_emit,gamma)
        zdelta_depth = zdelta_2016(zdesc_comp,zatm_comp,zemit_comp,EI_iceno,alpha_desc,alpha_atm,alpha_emit,gamma_fit=0)
    else:
        # 2025 version
        zdelta_comp = zdelta_2025(zdesc_comp,zatm_comp,zemit_comp,N_form,a_eff,alpha_desc,alpha_atm,alpha_emit,gamma)
        zdelta_depth = zdelta_2025(zdesc_comp,zatm_comp,zemit_comp,N_form,a_eff,alpha_desc,alpha_atm,alpha_emit,gamma_fit=0)
        
    fNs = arctangent_function(zdelta_comp/100,beta1,alpha0,beta0)
    fNs_H = arctangent_function(zdelta_depth/100,beta1,alpha0,beta0)
    if fNs_H > 1.0:
        fNs_H = 1.0
    elif fNs_H < 0.0:
        fNs_H = 0.0

    if fNs_H < x_s:
        H = zdesc_comp * eta1 * fNs_H
    else:
        H = zdesc_comp * (eta2 * fNs_H + (eta1 - eta2) * x_s)
    
    return fNs, H, zdelta_comp, zdesc_comp, zatm_comp, zemit_comp

def zatm_fit(s_i,T):
    # inputs: 
    # s_i: ambient supersaturation s_i (in 1)
    # T: ambient temperature (in K)
    # output: zatm based on fit formula (Eq. A2 in LU2025)
    a,b,c = 607.46, 0.897, 2.225
    return a * (s_i**b) * (T/205)**c

def zemit_fit(rho_emit,T):
    # inputs: 
    # rho_emit: emitted water vapor concentration (in kg/m^3)
    # T: ambient temperature (in K)
    # output: zemit based on fit formula (Eq. A3 in LU2025)
    a,b,c,d,e = 1106.6, 0.678, 0.0807, 0.0116, 0.000428
    return a * ((rho_emit*1e5)**(b+d*(T-205))) * np.exp(-(c+e*(T-205))* (T-205))

#example call
contrail_param(2,217,120,2.8e14,0.0115,60.3,15.0,520,0.0)