! Numerical implementation of the contrail depth and ice crystal number parametrisation as described in Unterstrasser, ACP, 2016
! written by Simon Unterstrasser, DLR Oberpfaffenhofen
! If you want to use the program, please contact: simon.unterstrasser@dlr.de
! The actual parametrisation is coded in the subroutine contrail_param.
! The main program contrail_properties contains various applications of the parametrisation and gives several examples of how to call the subroutine.
! The examples mainly reflect applications discussed in section 4 of Unterstrasser, ACP, 2016

PROGRAM contrail_parametrisation

    ! example calls of the parametrisation routine "contrail_param"
    ! compilation at DLR with ifort contrail_properties.f90
    IMPLICIT NONE
    INTEGER :: iversion

    DO iversion=1,3
        CALL Application_TableA2_U2016(iversion)
        CALL Application_Section4_1a_U2016(iversion)
        CALL Application_Section4_1b_U2016(iversion)
        CALL Application_Section4_2_U2016(iversion)
    ENDDO
    CALL Application_Table_LU2025(2)
END

SUBROUTINE Application_Table_LU2025(iversion)
    ! ----------------------------------------------------------------------------
    ! Example I a:
    ! ----------------------------------------------------------------------------

    !  Temp and Rhi-values are chosen as in selected lines of Table A1 of the corresponding publication Lottermoser & Unterstrasser (2025).
    IMPLICIT NONE

    INTEGER, INTENT(in) :: iversion
    INTEGER, PARAMETER::   nr=24
    REAL, DIMENSION(nr)::  Temp,RHi,EI_iceno,NBV,b,WVemit,Gamma0,EI_H2O
    REAL::H,N,N_form, zatm,zemit,zdesc
    REAL,DIMENSION(5):: fNs
    INTEGER:: i, i_scaling

    CHARACTER(len=13), DIMENSION (3) :: suffix_vec =(/'_param2016bis', '_param2025bis','_param2025fit'/)

    NBV=1.15e-2
    ! 12 simulations with A350/B777
    b(1:12)=60.3
    Gamma0(1:12)=520
    Temp(1:12) = (/217.,217.,217.,217.,  &
                   225.,225.,225.,225.,  &
                   233.,233.,233.,233./)
    RHi(1:12) =  (/1.2 ,1.1 ,1.2 ,1.1 ,  &
                   1.2 ,1.1 ,1.2 ,1.1 ,  &
                   1.2 ,1.1, 1.2 ,1.1 /)
    WVemit((/1,2,5,6,9,10/))     = 15e-3
    WVemit(2+(/1,2,5,6,9,10/))   = 15e-3*2.57
    EI_H2O((/1,2,5,6,9,10/))     = 1.26
    EI_H2O(2+(/1,2,5,6,9,10/))   = 8.94
    EI_iceno((/1,2,5,6,9,10/))   = 2.8e14
    EI_iceno(2+(/1,2,5,6,9,10/)) = 7.81e14

    ! 12 simulations with A320/B737
    b(13:24)=34.4
    Gamma0(13:24)=240
    Temp(13:24) = Temp(1:12)
    RHi(13:24) = RHi(1:12)
    WVemit(13:24) = WVemit(1:12)*(3.7/15.)
    EI_H2O(13:24) = EI_H2O(1:12)
    EI_iceno(13:24) = EI_iceno(1:12)
    OPEN (13,FILE='data_block_Table_LU2024'//TRIM(suffix_vec(iversion)) //'.txt')

    WRITE (13,*) 'Temp, RHi, fNs/%                            , zdesc/m, zemit/m, zatm/m'
    DO i=1,nr
        DO i_scaling=1,5
            CALL contrail_param(iversion, Temp(i),RHi(i),EI_iceno(i)*10.**(i_scaling-3),NBV(i),b(i),WVemit(i),Gamma0(i),&
            EI_H2O(i),H,N,N_form, zatm,zemit,zdesc)
            fNs(i_scaling)= N/N_form*100.
        ENDDO
        WRITE (13,'(2I5,5F7.1,I5,2I10)') INT(Temp(i)),INT(RHi(i)*100), fNs, INT(zdesc), INT(zemit), INT(zatm)
    ENDDO

    CLOSE (13)

! note that the listed values of fNs in the table (there denoted with a hat) can differ slightly from the values listed here.
! This is only due to round-off errors
END SUBROUTINE Application_Table_LU2025

SUBROUTINE Application_TableA2_U2016(iversion)
    ! ----------------------------------------------------------------------------
    ! Example I:
    ! ----------------------------------------------------------------------------

    !  Temp and Rhi-values are chosen as in Block 1 simulation in Table A2 of the corresponding publication.
    IMPLICIT NONE

    INTEGER, INTENT(in) :: iversion
    INTEGER, PARAMETER::   nr=17
    REAL, DIMENSION(nr)::  Temp,RHi,EI_iceno,NBV,b,WVemit,Gamma0
    REAL::H,N,N_form, zatm,zemit,zdesc
    REAL:: fNs
    INTEGER:: i

    CHARACTER(len=13), DIMENSION (3) :: suffix_vec =(/'_param2016bis', '_param2025bis','_param2025fit'/)
    Temp = (/209.,209.,212.,212.,212.,217.,217.,217.,217.,217.,222.,222.,222.,222.,225.,225.,225./)
    RHi=   (/1.0 ,1.2 ,1.0 ,1.2 ,1.4 ,1.0 ,1.1,1.2 ,1.3,1.4,1.0 ,1.1,1.2 ,1.4,1.1,1.2 ,1.3/)
    EI_iceno= 2.8e14
    NBV=1.15e-2
    b=60.9
    WVemit=15e-3
    Gamma0=520

    OPEN (13,FILE='data_block_TableA2_U2016'//TRIM(suffix_vec(iversion)) //'.txt')


    WRITE (13,*) 'Temp, RHi, fNs/%, H/m, zdesc/m, zemit/m, zatm/m'
    DO i=1,nr
        CALL contrail_param(iversion, Temp(i),RHi(i),EI_iceno(i),NBV(i),b(i),WVemit(i),Gamma0(i), 0.0,&
            H,N,N_form, zatm,zemit,zdesc)
            fNs= N/N_form*100.
        WRITE (13,'(2I5,F7.1,2I5,2I10)') INT(Temp(i)),INT(RHi(i)*100), fNs, INT(H), INT(zdesc), INT(zemit), INT(zatm)
    ENDDO

    CLOSE (13)

! note that the listed values of fNs and  H in Table A2 (there denoted with a hat) can differ slightly from the values listed here.
! This is only due to round-off errors
END SUBROUTINE APPLICATION_TABLEA2_U2016

SUBROUTINE Application_Section4_1a_U2016(iversion)

    ! ----------------------------------------------------------------------------
    ! Example II:
    ! ----------------------------------------------------------------------------
    !  described in Section 4.1 of the corresponding publication

    ! Check Reduction of EI_iceno by factor 10
    ! What is the effective reduction after the vortex phase
    ! RHi runs from 100% to 140% with increments of 5%
    ! Temp runs 210K to 226K with increments of 2K
    ! NBV runs from 0.6.e-2s^-1 to 1.4e-2s^-1 with increments of 0.1e-2s^-1
    ! b runs from 20m to 84m with increments of 8m

    IMPLICIT NONE
    INTEGER, INTENT(in) :: iversion
    REAL:: H,N,N_form, zatm,zemit,zdesc
    INTEGER:: i,j,k,l,m
    REAL::  Temp_2,RHi_2,NBV_2,b_2,EI_2
    REAL::N15=0.,N14=0.,N13=0.,N15form=0.,N14form=0.,N13form=0.
    REAL, PARAMETER :: nr_cases_inv = 1./(9*9*9*9)
    CHARACTER(len=13), DIMENSION (3) :: suffix_vec =(/'_param2016bis', '_param2025bis','_param2025fit'/)

    DO i=0,8
    RHi_2=1.0+0.05*i
        DO j=0,8
            Temp_2=210+j*2
            DO k=0,8
                NBV_2=0.1e-2*(6+k)
                DO l=0,8
                    b_2=20+8*l
                    EI_2=1e15
                    call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0,H,N,N_form, zatm,zemit,zdesc)
                    N15 = N15 + N
                    N15form = N15form + N_form

                    EI_2=1e14
                    call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0,H,N,N_form, zatm,zemit,zdesc)
                    N14 = N14 + N
                    N14form = N14form + N_form

                    EI_2=1e13
                    call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0,H,N,N_form, zatm,zemit,zdesc)
                    N13 = N13 + N
                    N13form = N13form + N_form
                ENDDO
            ENDDO
        ENDDO
    ENDDO
    N15=N15*nr_cases_inv
    N14=N14*nr_cases_inv
    N13=N13*nr_cases_inv
    N15form=N15form*nr_cases_inv
    N14form=N14form*nr_cases_inv
    N13form=N13form*nr_cases_inv

    OPEN (13,FILE='data_factor10_reduction_U2016'//TRIM(suffix_vec(iversion)) //'.txt')

    WRITE (13,'(A30,3E13.4)') 'EI15',N15form,N15,N15/N15form
    WRITE (13,'(A30,3E13.4)') 'EI14',N14form,N14,N14/N14form
    WRITE (13,'(A30,3E13.4)') 'EI13',N13form,N13,N13/N13form
    WRITE (13,'(A30,3F10.3)') 'N15/N14,N15/N13,N14/N13',N15/N14,N15/N13,N14/N13
    CLOSE (13)

END SUBROUTINE Application_Section4_1a_U2016

SUBROUTINE Application_Section4_1b_U2016(iversion)
  ! ----------------------------------------------------------------------------
  ! Example III:
  ! ----------------------------------------------------------------------------
  !  described in Section 4.1 of the corresponding publication
    IMPLICIT NONE
    INTEGER, INTENT(in) :: iversion
    REAL::H,N,N_form, zatm,zemit,zdesc
    INTEGER:: i,j,k,l,m
    REAL::  Temp_2,RHi_2,NBV_2,b_2,EI_2
    REAL,DIMENSION(0:8,0:8):: Nform_RHiEIiceno=0,N_RHiEIiceno=0,meanf_RHiEIiceno=0,Nsquare_RHiEIiceno=0
    REAL,DIMENSION(0:8,0:8):: Nform_TcaEIiceno=0,N_TcaEIiceno=0,meanf_TcaEIiceno=0,Nsquare_TcaEIiceno=0
    REAL,DIMENSION(0:8,0:8):: Nform_NBVEIiceno=0,N_NBVEIiceno=0,meanf_NBVEIiceno=0,Nsquare_NBVEIiceno=0
    REAL,DIMENSION(0:8,0:8):: Nform_spanEIiceno=0,N_spanEIiceno=0,meanf_spanEIiceno=0,Nsquare_spanEIiceno=0
    REAL, PARAMETER :: nr_cases_inv = 1./(9*9*9)
    CHARACTER(len=13), DIMENSION (3) :: suffix_vec =(/'_param2016bis', '_param2025bis','_param2025fit'/)

    DO i=0,8
    RHi_2=1.0+0.05*i
        DO j=0,8
            Temp_2=210+j*2
            DO k=0,8
                NBV_2=0.1e-2*(6+k)
                DO l=0,8
                    b_2=20+8*l
                    DO m=0,8
                        EI_2=1e12*10.0**(0.5*m)
                        call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0, H,N,N_form, zatm,zemit,zdesc)

                        Nform_RHiEIiceno(i,m) = Nform_RHiEIiceno(i,m) + N_form
                        N_RHiEIiceno(i,m) = N_RHiEIiceno(i,m) + N
                        Nsquare_RHiEIiceno(i,m) = Nsquare_RHiEIiceno(i,m) + N*N
                        meanf_RHiEIiceno(i,m) = meanf_RHiEIiceno(i,m) + N/N_form

                        Nform_TcaEIiceno(j,m) = Nform_TcaEIiceno(j,m) + N_form
                        N_TcaEIiceno(j,m) = N_TcaEIiceno(j,m) + N
                        Nsquare_TcaEIiceno(j,m) = Nsquare_TcaEIiceno(j,m) + N*N
                        meanf_TcaEIiceno(j,m) = meanf_TcaEIiceno(j,m) + N/N_form

                        Nform_NBVEIiceno(k,m) = Nform_NBVEIiceno(k,m) + N_form
                        N_NBVEIiceno(k,m) = N_NBVEIiceno(k,m) + N
                        Nsquare_NBVEIiceno(k,m) = Nsquare_NBVEIiceno(k,m) + N*N
                        meanf_NBVEIiceno(k,m) = meanf_NBVEIiceno(k,m) + N/N_form

                        Nform_spanEIiceno(l,m) = Nform_spanEIiceno(l,m) + N_form
                        N_spanEIiceno(l,m) = N_spanEIiceno(l,m) + N
                        Nsquare_spanEIiceno(l,m) = Nsquare_spanEIiceno(l,m) + N*N
                        meanf_spanEIiceno(l,m) = meanf_spanEIiceno(l,m) + N/N_form
                    ENDDO
                ENDDO
            ENDDO
        ENDDO
    ENDDO

    Nform_RHiEIiceno  = Nform_RHiEIiceno*nr_cases_inv
    N_RHiEIiceno      = N_RHiEIiceno*nr_cases_inv
    Nsquare_RHiEIiceno= Nsquare_RHiEIiceno*nr_cases_inv
    meanf_RHiEIiceno  = meanf_RHiEIiceno*nr_cases_inv

    Nform_TcaEIiceno  = Nform_TcaEIiceno*nr_cases_inv
    N_TcaEIiceno      = N_TcaEIiceno*nr_cases_inv
    Nsquare_TcaEIiceno= Nsquare_TcaEIiceno*nr_cases_inv
    meanf_TcaEIiceno  = meanf_TcaEIiceno*nr_cases_inv

    Nform_NBVEIiceno  = Nform_NBVEIiceno*nr_cases_inv
    N_NBVEIiceno      = N_NBVEIiceno*nr_cases_inv
    Nsquare_NBVEIiceno= Nsquare_NBVEIiceno*nr_cases_inv
    meanf_NBVEIiceno  = meanf_NBVEIiceno*nr_cases_inv

    Nform_spanEIiceno  = Nform_spanEIiceno*nr_cases_inv
    N_spanEIiceno      = N_spanEIiceno*nr_cases_inv
    Nsquare_spanEIiceno= Nsquare_spanEIiceno*nr_cases_inv
    meanf_spanEIiceno  = meanf_spanEIiceno*nr_cases_inv

    !OPEN (13,FILE='Datenmatrix_N_RHi_EIiceno.txt')
    OPEN (13,FILE='data_matrix_N_RHi_EIiceno'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(81E13.4)')  N_RHiEIiceno
    WRITE (13,'(81E13.4)')  Nform_RHiEIiceno
    WRITE (13,'(81E13.4)')  Nsquare_RHiEIiceno
    WRITE (13,'(81E13.4)')  meanf_RHiEIiceno
    CLOSE (13)

    !OPEN (13,FILE='Datenmatrix_N_Tca_EIiceno.txt')
    OPEN (13,FILE='data_matrix_N_Tca_EIiceno'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(81E13.4)')  N_TcaEIiceno
    WRITE (13,'(81E13.4)')  Nform_TcaEIiceno
    WRITE (13,'(81E13.4)')  Nsquare_TcaEIiceno
    WRITE (13,'(81E13.4)')  meanf_TcaEIiceno
    CLOSE (13)

    !OPEN (13,FILE='Datenmatrix_N_NBV_EIiceno.txt')
    OPEN (13,FILE='data_matrix_N_NBV_EIiceno'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(81E13.4)')  N_NBVEIiceno
    WRITE (13,'(81E13.4)')  Nform_NBVEIiceno
    WRITE (13,'(81E13.4)')  Nsquare_NBVEIiceno
    WRITE (13,'(81E13.4)')  meanf_NBVEIiceno
    CLOSE (13)

    !OPEN (13,FILE='Datenmatrix_N_span_EIiceno.txt')
    OPEN (13,FILE='data_matrix_N_span_EIiceno'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(81E13.4)')  N_spanEIiceno
    WRITE (13,'(81E13.4)')  Nform_spanEIiceno
    WRITE (13,'(81E13.4)')  Nsquare_spanEIiceno
    WRITE (13,'(81E13.4)')  meanf_spanEIiceno
    CLOSE (13)

END SUBROUTINE Application_Section4_1b_U2016

SUBROUTINE Application_Section4_2_U2016(iversion)
    ! ----------------------------------------------------------------------------
    ! Example IV:
    ! ----------------------------------------------------------------------------
    !  described in Section 4.2 of the corresponding publication

    !Check Sensitivity to the input parameters
    !What is the effective reduction after the vortex phase
    ! RHi runs from 100% to 140% with increments of 5%
    ! Temp runs 210K to 226K with increments of 2K
    ! NBV runs from 0.6.e-2s^-1 to 1.4e-2s^-1 with increments of 0.1e-2s^-1
    ! b runs from 20m to 84m with increments of 8m
    INTEGER, INTENT(in) :: iversion
    REAL,DIMENSION(0:8,0:8,0:8,0:8)::N15vec,N14vec,N15formvec,N14formvec,H15vec,H14vec,Nconc15vec,Nconc14vec
    REAL,DIMENSION(0:8):: N15RHi, N15formRHi, N15Temp, N15formTemp, N15NBV, N15formNBV, N15span, N15formspan
    REAL,DIMENSION(0:8):: H15RHi, H15Temp, H15NBV, H15span, Nconc15RHi, Nconc15Temp, Nconc15NBV, Nconc15span
    REAL,DIMENSION(0:8):: N14RHi, N14formRHi, N14Temp, N14formTemp, N14NBV, N14formNBV, N14span, N14formspan
    REAL,DIMENSION(0:8):: H14RHi, H14Temp, H14NBV, H14span, Nconc14RHi, Nconc14Temp, Nconc14NBV, Nconc14span
    INTEGER:: i,j,k,l,m
    REAL::  Temp_2,RHi_2,NBV_2,b_2,EI_2
    REAL::H,N,N_form, zatm,zemit,zdesc
    REAL, PARAMETER :: nr_cases_inv = 1./(9*9*9)
    CHARACTER(len=13), DIMENSION (3) :: suffix_vec =(/'_param2016bis', '_param2025bis','_param2025fit'/)

     DO i=0,8
       RHi_2=1.0+0.05*i
       DO j=0,8
         Temp_2=210+j*2
         DO k=0,8
           NBV_2=0.1e-2*(6+k)
           DO l=0,8
             b_2=20+8*l
             EI_2=1e15
             call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0, H,N,N_form, zatm,zemit,zdesc)
             N15vec(i,j,k,l)= N
             N15formvec(i,j,k,l)= N_form
             H15vec(i,j,k,l)= H
              if (N.gt.0.0.AND.H.gt.0) Nconc15vec(i,j,k,l)= N/(H *0.63*b_2)*1e-6
    !         if (N.gt.0.0.AND.H.gt.0) Nconc15vec(i,j,k,l)= N/(H *36)*1e-6

             EI_2=1e14
             call contrail_param(iversion, Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,0.0, H,N,N_form, zatm,zemit,zdesc)
             N14vec(i,j,k,l)= N
             N14formvec(i,j,k,l)= N_form
             H14vec(i,j,k,l)= H
              if (N.gt.0.0.AND.H.gt.0) Nconc14vec(i,j,k,l)= N/(H *0.63*b_2) *1e-6
    !         if (N.gt.0.0.AND.H.gt.0) Nconc14vec(i,j,k,l)= N/(H *36) *1e-6

           ENDDO
         ENDDO
       ENDDO
     ENDDO

    N15RHi = SUM(SUM(SUM(N15vec,4),3),2)*nr_cases_inv
    N15formRHi = SUM(SUM(SUM(N15formvec,4),3),2)*nr_cases_inv
    H15RHi = SUM(SUM(SUM(H15vec,4),3),2)*nr_cases_inv
    Nconc15RHi = SUM(SUM(SUM(Nconc15vec,4),3),2)*nr_cases_inv

    N15Temp = SUM(SUM(SUM(N15vec,4),3),1)*nr_cases_inv
    N15formTemp = SUM(SUM(SUM(N15formvec,4),3),1)*nr_cases_inv
    H15Temp = SUM(SUM(SUM(H15vec,4),3),1)*nr_cases_inv
    Nconc15Temp = SUM(SUM(SUM(Nconc15vec,4),3),1)*nr_cases_inv

    N15NBV = SUM(SUM(SUM(N15vec,4),2),1)*nr_cases_inv
    N15formNBV = SUM(SUM(SUM(N15formvec,4),2),1)*nr_cases_inv
    H15NBV = SUM(SUM(SUM(H15vec,4),2),1)*nr_cases_inv
    Nconc15NBV = SUM(SUM(SUM(Nconc15vec,4),2),1)*nr_cases_inv

    N15span = SUM(SUM(SUM(N15vec,3),2),1)*nr_cases_inv
    N15formspan = SUM(SUM(SUM(N15formvec,3),2),1)*nr_cases_inv
    H15span = SUM(SUM(SUM(H15vec,3),2),1)*nr_cases_inv
    Nconc15span = SUM(SUM(SUM(Nconc15vec,3),2),1)*nr_cases_inv

    OPEN (13,FILE='param_sens'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(9E13.4)') N15RHi
    WRITE (13,'(9E13.4)') N15formRHi
    WRITE (13,'(9E13.4)') N15Temp
    WRITE (13,'(9E13.4)') N15formTemp
    WRITE (13,'(9E13.4)') N15NBV
    WRITE (13,'(9E13.4)') N15formNBV
    WRITE (13,'(9E13.4)') N15span
    WRITE (13,'(9E13.4)') N15formspan

    WRITE (13,'(9E13.4)') H15RHi
    WRITE (13,'(9E13.4)') H15Temp
    WRITE (13,'(9E13.4)') H15NBV
    WRITE (13,'(9E13.4)') H15span

    WRITE (13,'(9E13.4)') Nconc15RHi
    WRITE (13,'(9E13.4)') Nconc15Temp
    WRITE (13,'(9E13.4)') Nconc15NBV
    WRITE (13,'(9E13.4)') Nconc15span

    N14RHi = SUM(SUM(SUM(N14vec,4),3),2)*nr_cases_inv
    N14formRHi = SUM(SUM(SUM(N14formvec,4),3),2)*nr_cases_inv
    H14RHi = SUM(SUM(SUM(H14vec,4),3),2)*nr_cases_inv
    Nconc14RHi = SUM(SUM(SUM(Nconc14vec,4),3),2)*nr_cases_inv

    N14Temp = SUM(SUM(SUM(N14vec,4),3),1)*nr_cases_inv
    N14formTemp = SUM(SUM(SUM(N14formvec,4),3),1)*nr_cases_inv
    H14Temp = SUM(SUM(SUM(H14vec,4),3),1)*nr_cases_inv
    Nconc14Temp = SUM(SUM(SUM(Nconc14vec,4),3),1)*nr_cases_inv

    N14NBV = SUM(SUM(SUM(N14vec,4),2),1)*nr_cases_inv
    N14formNBV = SUM(SUM(SUM(N14formvec,4),2),1)*nr_cases_inv
    H14NBV = SUM(SUM(SUM(H14vec,4),2),1)*nr_cases_inv
    Nconc14NBV = SUM(SUM(SUM(Nconc14vec,4),2),1)*nr_cases_inv

    N14span = SUM(SUM(SUM(N14vec,3),2),1)*nr_cases_inv
    N14formspan = SUM(SUM(SUM(N14formvec,3),2),1)*nr_cases_inv
    H14span = SUM(SUM(SUM(H14vec,3),2),1)*nr_cases_inv
    Nconc14span = SUM(SUM(SUM(Nconc14vec,3),2),1)*nr_cases_inv

    WRITE (13,'(9E13.4)') N14RHi
    WRITE (13,'(9E13.4)') N14formRHi
    WRITE (13,'(9E13.4)') N14Temp
    WRITE (13,'(9E13.4)') N14formTemp
    WRITE (13,'(9E13.4)') N14NBV
    WRITE (13,'(9E13.4)') N14formNBV
    WRITE (13,'(9E13.4)') N14span
    WRITE (13,'(9E13.4)') N14formspan

    WRITE (13,'(9E13.4)') H14RHi
    WRITE (13,'(9E13.4)') H14Temp
    WRITE (13,'(9E13.4)') H14NBV
    WRITE (13,'(9E13.4)') H14span

    WRITE (13,'(9E13.4)') Nconc14RHi
    WRITE (13,'(9E13.4)') Nconc14Temp
    WRITE (13,'(9E13.4)') Nconc14NBV
    WRITE (13,'(9E13.4)') Nconc14span
    CLOSE (13)

    OPEN (13,FILE='data_matrix_N15_N14_Nconc15_Nconc14_H'//TRIM(suffix_vec(iversion)) //'.txt')
    WRITE (13,'(9E13.4)')  N15vec
    WRITE (13,'(9E13.4)')  N14vec
    WRITE (13,'(9E13.4)')  Nconc15vec
    WRITE (13,'(9E13.4)')  Nconc14vec
    WRITE (13,'(9E13.4)')  H15vec
    CLOSE (13)

END SUBROUTINE Application_Section4_2_U2016

SUBROUTINE contrail_param(iversion, &
                          Temp,RHi,EI_iceno,NBV,b,WVemit,Gamma0, EI_H2O,&
                          H,N,N_form, &
                          zatm,zemit,zdesc)
! all inpupt and output variables are scalars
! output variables:
! H         contrail height after the vortex phase in m
! N         number of ice crystals per meter (of flight path) after the vortex phase
! N_form    initial number of ice crystals per meter (of flight path)
! z_atm     atmospheric length scale
! z_emit    emission length scale
! z_desc    descent length scale

! input variables:
! Temp      temperature in K
! RHi       relative humidity w.r.t. ice, 1.0=100%
! EI_iceno      number of ice crystal per kg fuel
! EI_H2O    emission index of water vapour in kg per kg fuel (if unspecified, set it to 0)
! NBV       Brunt Väisäla frequency in s^-1
! b         wing span in m
! WVemit    water vapor emission in kg/m, optional (if unspecified, set it to 0)
! Gamma0    initial circulation in m^2/s, optional (if unspecified, set it to 0)
! iversion  specifies which version of the parametrisation is used
!           = 1 version as published in Unterstrasser (2016)
!           = 2 version as published in Lottermoser & Unterstrasser (2024, in rev.)
!           = 3 identical to iversion=2, except that fit function for z_emit and z_atm are used (instead of solving a non-linear equation with a bisection approach)
! This program computes approximations of H and N of ~5-minute old contrails for the given input parameters
! The approximations are based on the parametrisations presented in Unterstrasser, 2016, ACP, "Properties of young contrails..."
! The parametrisation was updated in Lottermoser & Unterstrasser (2024, in rev.)
! Note: If WVemit and/or Gamma0 are not available, simple estimates based on b are used.
! assumptions
! cruise speed U =235m/s

    IMPLICIT NONE

    REAL, INTENT(in) :: Temp,RHi,EI_iceno,NBV,b,WVemit,Gamma0,EI_H2O
    INTEGER, INTENT(in) :: iversion
    REAL, INTENT(out) :: N, H, N_form
    REAL :: WVemitn,Gamma0n,EI_H2On
    REAL :: zdesc, zatm, zemit
    REAL :: pi = 3.14
    REAL :: rp, Ap, prefactor_Ap_formula ! plume radius rp (in m), plume area (in m^2)
    REAL :: rho_emit ! surplus WV concentration through emission

    ! fit coefficients
    REAL :: beta0, beta1, alpha0, alpha_desc, alpha_atm, alpha_emit
    REAL :: gamma_atm, gamma_emit
    REAL :: x_s, eta1, eta2

    REAL :: EI_iceno_ref = 2.8e14  !(number of ice crystals per kg fuel of reference case; old approach)
    REAL :: N0_ice_ref = 3.38e12   !(number of ice crystals per m of flight distance of reference case; new approach)
    REAL :: bspan_ref = 60.3      !(wing span in m of reference case , required to calculate plume area of reference case; new approach)
    REAL :: rp_ref, Ap_ref, Nconc_ref, Nconc
    REAL :: alpha_atm_corr,alpha_emit_corr,Psi_star_inv
    REAL :: fNs, fNs_H, zDelta_N, zDelta_H
    REAL,EXTERNAL:: zbuffer2016, zbuffer2025, zatm_fit,zemit_fit
    !compute the three length scales zdesc, zatm and zemit
    !   PRINT*, 'Temp,RHi,EI,NBV,b,WVemit,Gamma0',Temp,RHi,EI,NBV,b,WVemit,Gamma0

    ! contrail height fitting coefficients
    x_s=0.2; eta1=6.0; eta2=0.15

    SELECT CASE (iversion)
        CASE (1)
            beta0 = 0.45; beta1 = 1.19
            alpha0 = -1.35; alpha_desc=0.6; alpha_atm=1.7; alpha_emit=1.15
            gamma_atm = 0.18; gamma_emit = 0.18
            prefactor_Ap_formula = 4. ! the formula for A_p in U2016 (Eq.A7) was not well-defined
        CASE (2:3)
            beta0 = 0.42; beta1 = 1.31
            alpha0 = -1.00; alpha_desc=0.49; alpha_atm=1.27; alpha_emit=0.42
            gamma_atm = 0.16; gamma_emit = 0.16
            prefactor_Ap_formula = 2.
        CASE DEFAULT
            PRINT*,"incorrect iversion value"
    END SELECT


    ! if parameter is unspecified, fall back to default value or empirical relationship
    Gamma0n=Gamma0
    if (Gamma0n.eq.0) Gamma0n = 10.*b-70.

    WVemitn=WVemit
    if (WVemitn.eq.0.) then
        WVemitn = 3.1250e-06 *b*b
    endif

    EI_H2On = EI_H2O
    if (EI_H2On.eq.0) EI_H2On = 1.26
    !compute zdesc
    zdesc =  sqrt(8*Gamma0n/(pi*NBV))
    !compute concentration delta by WV emission
     ! compute rho_emit = WV/Ap
    ! compute Ap
    rp= 1.5+0.314*b
    Ap=prefactor_Ap_formula*pi*rp*rp
    rho_emit = WVemitn/Ap ! units kg/m^3
    !PRINT*,'rho_emit', rho_emit
    N_form = WVemitn/EI_H2On*EI_iceno

    !compute zatm
    SELECT CASE (iversion)
        CASE (1)
            zatm = zbuffer2016(Temp,RHi,0.0)
            zemit = zbuffer2016(Temp,1.0,rho_emit)
        CASE (2)
            zatm = zbuffer2025(Temp,RHi,0.0)
            zemit = zbuffer2025(Temp,1.0,rho_emit)
        CASE(3)
            zatm = zatm_fit(Temp,RHi)
            zemit = zemit_fit(Temp,rho_emit)
    END SELECT

!  PRINT*,'rp,Ap,rho_emit',rp,Ap,rho_emit

! compute zDelta
    SELECT CASE (iversion)
        CASE (1)
            Psi_star_inv  = EI_iceno_ref / EI_iceno

        CASE(2:3)
            rp_ref= 1.5+0.314*bspan_ref
            Ap_ref=prefactor_Ap_formula*pi*rp_ref*rp_ref

            Nconc_ref = N0_ice_ref/Ap_ref
            Nconc = N_form/Ap
            Psi_star_inv = Nconc_ref / Nconc
    END SELECT

    alpha_atm_corr = alpha_atm * Psi_star_inv**(gamma_atm)
    alpha_emit_corr = alpha_emit * Psi_star_inv**(gamma_emit)

  zDelta_N = alpha_atm_corr * zatm + alpha_emit_corr * zemit - alpha_desc * zdesc
  zDelta_H = alpha_atm      * zatm + alpha_emit      * zemit - alpha_desc * zdesc
 ! PRINT*,' alpha_atm_corr, alpha_emit_corr , alpha_desc',alpha_atm_corr, alpha_emit_corr , alpha_desc
 ! PRINT*,- alpha_desc * zdesc, alpha_atm_corr * zatm , alpha_emit_corr * zemit
 ! PRINT*,'AA', zDelta_N,  alpha0+zDelta_N*0.01

! compute survival fraction fNs and total ice crystal number

  fNs = beta0 + beta1/pi * atan(alpha0+zDelta_N*0.01)
  if (fNs.gt.1.) fNs = 1.0
  if (fNs.lt.0.) fNs = 0.0
  N = N_form * fNs

! compute contrail depth
  fNs_H = beta0 + beta1/pi * atan(alpha0+zDelta_H*0.01)
  if (fNs_H.gt.1.) fNs_H = 1.0
  if (fNs_H.lt.0.) fNs_H = 0.0
  if (fNs_H.lt.x_s) then
     H = zdesc * eta1*fNs_H
  else
     H = zdesc * (eta2*fNs_H + (eta1-eta2)*x_s)
  endif

END SUBROUTINE contrail_param

FUNCTION esat(Temp)
    !input T in K
    ! output saturation pressure over ice in Pa
    REAL :: Temp
    REAL :: esat
    esat = exp(9.550426-5723.265/Temp+3.53068*alog(Temp)-0.00728332*Temp)  ! in Pa
END FUNCTION esat

REAL FUNCTION zbuffer2016(Temp,RHi,rho_emit)
! input variables:
! temperature Temp
! relative humidity RHi
! emitted surplus water vapor concentration rho_emit
! output: zatm (if rho_emit is 0)
!         zemit (if RHi=1.0)

! solve euqation:
! 1/R_v * (1+s_i) * e_s(T_CA)/T_CA +  rho_emit  = 1/R_v * e_s(T_CA + T_buffer) / (T_CA + T_buffer)
! z_buffer = T_buffer / Gamma_d
    implicit none
    REAL, INTENT(in)::Temp,RHi,rho_emit
    REAL :: R_v=  461.5  !J /(K * kg)
    REAL :: si,rhov_left, rho_tot
    REAL :: dT(2)
    REAL :: T,x
    REAL,EXTERNAL:: esat
    INTEGER::n_iterations,i
    si= RHi-1.0
    rhov_left= (1+si)*esat(Temp)/(R_v*Temp) ! in kg/m^3
    rho_tot= rhov_left + rho_emit

    !bisection method
    dT(1)=0.0
    dT(2)=20.0

    n_iterations=19  ! -> epsilon = 10/(2^n_iterations) = 3.0518e-04 K in Temp

    DO i=0,n_iterations
        T=Temp+(dT(1)+dT(2))/2
        x= esat(T)/(R_v*T)
        if (x.GE.rho_tot) then
          dT(2)=(dT(1)+dT(2))/2
        else
          dT(1)=(dT(1)+dT(2))/2
        endif
    ENDDO
    if (dT(2).eq.20.0) then
       PRINT*,'Choose a larger initial value for dT(2)!!'
    endif
    !PRINT*,(dT(1)+dT(2))/2,rho_emit
    zbuffer2016 = ((dT(1)+dT(2))/2 / 9.8*1000.0  )
  END FUNCTION zbuffer2016

REAL FUNCTION zbuffer2025(Temp,RHi,rho_emit)
! input variables:
! temperature Temp
! relative humidity RHi
! emitted surplus water vapor concentration rho_emit
! output: zatm (if rho_emit is 0)
!         zemit (if RHi=1.0)

! solve euqation:
! 1/R_v * (1+s_i) * e_s(T_CA)/(T_CA^kappa) +  rho_emit/(T_CA^(kappa-1))  = 1/R_v * e_s(T_CA + T_buffer) / (T_CA + T_buffer)^kappa
! z_buffer = T_buffer / Gamma_d
    implicit none
    REAL, INTENT(in)::Temp,RHi,rho_emit
    REAL :: R_v=  461.5  !J /(K * kg)
    REAL :: si,rhov_left, rho_tot
    REAL :: dT(2)
    REAL :: T,x
    INTEGER::n_iterations,i
    REAL,EXTERNAL:: esat
    REAL :: kappa = 3.5, kappa_m1 = 2.5
    si= RHi-1.0
    rhov_left= (1+si)*esat(Temp)/(R_v*Temp**kappa) ! in kg/m^3
    rho_tot= rhov_left + rho_emit/Temp**kappa_m1

    !bisection method
    dT(1)=0.0
    dT(2)=20.0

    n_iterations=19  ! -> epsilon = 10/(2^n_iterations) = 3.0518e-04 K in Temp

    DO i=0,n_iterations
        T=Temp+(dT(1)+dT(2))/2
        x= esat(T)/(R_v*T**kappa)
        if (x.GE.rho_tot) then
          dT(2)=(dT(1)+dT(2))/2
        else
          dT(1)=(dT(1)+dT(2))/2
        endif
    ENDDO
    if (dT(2).eq.20.0) then
       PRINT*,'Choose a larger initial value for dT(2)!!'
    endif
    !PRINT*,(dT(1)+dT(2))/2,rho_emit
    zbuffer2025 = ((dT(1)+dT(2))/2 / 9.8*1000.0  )
  END FUNCTION zbuffer2025

REAL FUNCTION zatm_fit(Temp,RHi)
! input variables:
! temperature Temp in K
! relative humidity RHi in 1 (i.e. 1.1 = 110%)
! output: zatm in m

    REAL, PARAMETER :: a = 607.46, b = 0.897, c = 2.225
    zatm_fit =  a * ((RHi-1)**b) * (Temp/205)**c
END FUNCTION zatm_fit

REAL FUNCTION zemit_fit(Temp,rho_emit)
! input variables:
! temperature Temp in K
! rho_emit emitted WV concentration increase in kg/m^3

    REAL, PARAMETER :: a= 1106.6, b= 0.678, c= 0.0807
    REAL, PARAMETER :: d = 0.0116, e = 0.000428
    zemit_fit = a * ((rho_emit*1e5)**(b+d*(Temp-205))) * &
           exp(-(c+e*(Temp-205))* (Temp-205))
END FUNCTION zemit_fit