
C***********************************************************************
C   Portions of Models-3/CMAQ software were developed or based on      *
C   information from various groups: Federal Government employees,     *
C   contractors working on a United States Government contract, and    *
C   non-Federal sources (including research institutions).  These      *
C   research institutions have given the Government permission to      *
C   use, prepare derivative works, and distribute copies of their      *
C   work in Models-3/CMAQ to the public and to permit others to do     *
C   so.  EPA therefore grants similar permissions for use of the       *
C   Models-3/CMAQ software, but users are requested to provide copies  *
C   of derivative works to the Government without restrictions as to   *
C   use by others.  Users are responsible for acquiring their own      *
C   copies of commercial software associated with Models-3/CMAQ and    *
C   for complying with vendor requirements.  Software copyrights by    *
C   the MCNC Environmental Modeling Center are used with their         *
C   permissions subject to the above restrictions.                     *
C***********************************************************************

C:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
      SUBROUTINE READCSQY ( NWL, STWL, ENDWL, CS, QY )

C*********************************************************************
C
C  the subroutine readcsqy reads the absorption cross section/quantum
C     yield file(s).  The input data are
C
C     CS(nwl,NPHOTAB)        - absorption cross sections for NR species.
C     QY(nwl,NPHOTAB)        - quantum yields
C
C    S.Roselle  1/30/96  Subroutine created, uses generalized method
C                        for reading CS/QY data that ties in with the
C                        chemistry mechanism reader.
C    S.Roselle  7/25/96  Revised subroutine to call INTAVG, passing
C                        data type (e.g., point, centered,
C                        beginning, and ending data)
C
C*********************************************************************

      IMPLICIT NONE

      INCLUDE SUBST_RXCMMN       ! chemical mechamism reactions COMMON
      INCLUDE 'JVALPARMS.EXT'    ! jproc parameters

C...........PARAMETERS and their descriptions

      INTEGER      XSTAT1             ! I/O ERROR exit status
      PARAMETER  ( XSTAT1 = 1 )

      INTEGER      XSTAT2             ! Program ERROR exit status
      PARAMETER  ( XSTAT2 = 2 )

C...........ARGUMENTS and their descriptions

      REAL         STWL ( MXWL )       ! wavelength band lower limit
      REAL         ENDWL( MXWL )       ! wavelength band upper limit
      REAL         CS( MXWL, NPHOTAB )  ! output absorp. cross sections
      REAL         QY( MXWL, NPHOTAB )  ! output quantum yields

C...........LOCAL VARIABLES and their descriptions:

      CHARACTER*1  TYPE                ! cs/qy spectra type
      CHARACTER*16 PNAME               ! program name
      DATA         PNAME   / 'READCSQY' /
      CHARACTER*16 CQDIR               ! directory for CSQY data
      DATA         CQDIR   / 'CSQY' /
      CHARACTER*16 PHOTID              ! reaction id's
      CHARACTER*255 EQNAME
      CHARACTER*80 CQFILE              ! input filename buffer
      CHARACTER*80 MSG                 ! message
      DATA         MSG / '    ' /

      INTEGER      IWL                 ! wavelength index
      INTEGER      NWL                 ! # of wlbands
      INTEGER      NWLIN               ! # of wlbands (infile)
      INTEGER      IPHOT               ! reaction index
      INTEGER      CQUNIT              ! cross section/qy io unit
      INTEGER      IOST                ! io status
      INTEGER      LASTNB1
      INTEGER      LASTNB2

      REAL         FACTOR              ! multiplying factor for CS
      REAL         WLIN   ( MXWLIN )   ! wl for input cs/qy data
      REAL         CSIN ( MXWLIN )     ! raw absorption cross sections
      REAL         QYIN ( MXWLIN )     ! raw quantum yields
      REAL         CSOUT( MXWL )       ! integrated absorp. cross sect.
      REAL         QYOUT( MXWL )       ! integrated quantum yields

C...........EXTERNAL FUNCTIONS and their descriptions:

      INTEGER      JUNIT               ! used to get next IO unit #
      INTEGER      TRIMLEN             ! returns length of string

C*********************************************************************
C     begin body of subroutine READCSQY

C...get a unit number for CSQY files

      CQUNIT = JUNIT( )

C...loop over the number of reactions, reading each file

      DO 801 IPHOT = 1, NPHOTAB

C...open input file

        CQFILE = PHOTAB( IPHOT )
        LASTNB1 = TRIMLEN( CQFILE )
        CALL NAMEVAL ( CQDIR, EQNAME )
        LASTNB2 = TRIMLEN( EQNAME )
        CQFILE = EQNAME( 1:LASTNB2 ) // '/' // CQFILE( 1:LASTNB1 )

        OPEN( UNIT = CQUNIT,
     &        FILE = CQFILE,
     &        STATUS = 'OLD',
     &        IOSTAT = IOST )

C...check for open errors

        IF ( IOST .NE. 0) THEN
          MSG = 'Could not open ' // PHOTAB( IPHOT ) // ' data file'
          CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT1 )
        END IF

        WRITE( 6, 2001 ) CQUNIT, CQFILE

C...read photolysis subgroup id

        READ( CQUNIT, 1001, IOSTAT = IOST ) PHOTID

C...check for read errors

        IF ( IOST .NE. 0) THEN
          MSG = 'Errors occurred while reading PHOTID for ' //
     &           PHOTAB( IPHOT )
          CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT1 )
        END IF

C...get type of data (e.g. centered, beginning, ending, or point wavelen

101     CONTINUE

        READ( CQUNIT, 1003, IOSTAT = IOST ) TYPE

C...check for read errors

        IF ( IOST .NE. 0) THEN
          MSG = 'Errors occurred while reading TYPE for ' //
     &           PHOTAB( IPHOT )
          CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT1 )
        END IF

        IF ( TYPE .EQ. '!' ) GO TO 101

C...read the factor to multiply cross sectionS by

        READ( CQUNIT, 1005, IOSTAT = IOST ) FACTOR

C...check for read errors

        IF ( IOST .NE. 0) THEN
          MSG = 'Errors occurred while reading FACTOR for ' //
     &           PHOTAB( IPHOT )
          CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT1 )
        END IF

C...reinitialize arrays

        DO IWL = 1, MXWL
          WLIN( IWL ) = 0.0
          CSIN( IWL ) = 0.0
          QYIN( IWL ) = 0.0
        END DO

C...loop over the number of wavelengths and continue reading

        IWL = 0
201     CONTINUE

          IWL = IWL + 1
          READ( CQUNIT, *, IOSTAT = IOST ) WLIN( IWL ), CSIN( IWL ),
     &                                     QYIN( IWL )
          CSIN( IWL ) = CSIN( IWL ) * FACTOR

C...check for read errors

          IF ( IOST .GT. 0) THEN
            MSG = 'Errors occurred while reading WL,CS,QY for ' //
     &             PHOTAB( IPHOT )
            CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT1 )
          END IF

C...end loop if we reach EOF, otherwise continue looping

        IF ( IOST .EQ. 0 ) GO TO 201

C...adjust loop counter index index and close file

        NWLIN = IWL - 1
        CLOSE( CQUNIT )

        WRITE( 6, 2003 ) NWLIN

C...transform the cs data to the same wavelength intervals as
C...  the irradiance data.

        CALL INTAVG ( WLIN, CSIN, NWLIN, TYPE,
     &                STWL, ENDWL, CSOUT, NWL )

C...transform the qy data to the same wavelength intervals as
C...  the irradiance data.

        CALL INTAVG ( WLIN, QYIN, NWLIN, TYPE,
     &                STWL, ENDWL, QYOUT, NWL )

C...load output arrays with integrated data

        DO IWL = 1, NWL
          CS( IWL, IPHOT ) = CSOUT( IWL )
          QY( IWL, IPHOT ) = QYOUT( IWL )
        END DO

801   CONTINUE

C...formats

1001  FORMAT( A16 )
1003  FORMAT( A1 )
1005  FORMAT( /, 4X, F10.1 )

2001  FORMAT( 1X, '...Opening File on UNIT ', I2, /, 1X, A255 )
2003  FORMAT( 1X, '...Data for ', I4, ' wavelengths read from file',
     &        // )

      RETURN
      END
