
C***********************************************************************
C   Portions of Models-3/CMAQ software were developed or based on      *
C   information from various groups: Federal Government employees,     *
C   contractors working on a United States Government contract, and    *
C   non-Federal sources (including research institutions).  These      *
C   research institutions have given the Government permission to      *
C   use, prepare derivative works, and distribute copies of their      *
C   work in Models-3/CMAQ to the public and to permit others to do     *
C   so.  EPA therefore grants similar permissions for use of the       *
C   Models-3/CMAQ software, but users are requested to provide copies  *
C   of derivative works to the Government without restrictions as to   *
C   use by others.  Users are responsible for acquiring their own      *
C   copies of commercial software associated with Models-3/CMAQ and    *
C   for complying with vendor requirements.  Software copyrights by    *
C   the MCNC Environmental Modeling Center are used with their         *
C   permissions subject to the above restrictions.                     *
C***********************************************************************

      SUBROUTINE CHEM (CGRID, JDATE, JTIME, TSTEP)

C     BY: Lucas A. J. Bastien. Parts of this code (or all of it) may be
C     based on or inspired by pre-existing CMAQ and/or CMAQ adjoint
C     code, with possibly some sections copied without modification.
C
C     This subroutine performs the chemistry integration time step. It
C     is basically a wrapper around the subroutine INTEGRATE created by
C     KPP, with the SAPRC99 chemical mechanism and the Rosenbrock suite
C     of solvers.

C     ------------------------------------------------------------------

      USE HGRD_DEFN
      USE VGRD_DEFN
      USE saprc99ros_Precision
      USE saprc99ros_Global
      USE saprc99ros_Monitor
      USE saprc99ros_Initialize
      USE saprc99ros_Rates
      USE saprc99ros_Integrator
      USE CMAQKPP_SAPRC99_INTERFACE
      USE ADJPRM

      IMPLICIT NONE

!     Include file(s)

      INCLUDE SUBST_RXCMMN   ! Chemical mechanism information header
      INCLUDE SUBST_IOPARMS  ! I/O parameters definitions
      INCLUDE SUBST_IODECL   ! I/O definitions and declarations
      INCLUDE SUBST_GC_SPC   ! Gas phase species information
      INCLUDE SUBST_FILES_ID ! CMAQ files IDs
      INCLUDE SUBST_CONST    ! CMAQ constants

!     Dummy variable(s)

      REAL,    INTENT(IN), POINTER :: CGRID(:,:,:,:)
      INTEGER, INTENT(IN) :: JDATE
      INTEGER, INTENT(IN) :: JTIME
      INTEGER, INTENT(IN) :: TSTEP(2)

!     Local variable(s)

      LOGICAL, SAVE :: FIRSTIME = .TRUE.
      INTEGER, SAVE :: LOGDEV
      CHARACTER(16) :: PNAME = "CHEM"
      CHARACTER(96) :: MSG = ""
      REAL(kind=dp), SAVE :: TIN ! Start time for integration
      REAL(kind=dp) :: TOUT      ! End time for integration
      INTEGER :: MIDDATE, MIDTIME ! Date and time at mid time step
      INTEGER :: IC, IR, IL, IS, IPHOT ! Iteration variables
      INTEGER :: TSTEP2_SEC ! TSTEP(2) in seconds
      INTEGER :: STATUS           ! Error status returned by integrator
      LOGICAL :: LSUNLIGHT ! Logical specifying whether there is
                           ! sunlight
      REAL(KIND=dp), SAVE :: CST1 ! Constant
      REAL(KIND=dp), SAVE :: CST2 ! Constant
      INTEGER :: NDARK            ! Number of layer 1 cells in darkness
      REAL :: RJ(NCOLS, NROWS, NLAYS, NPHOTAB) ! J-values for each cell

!     Local variable(s) that control the Rosenbrock solver

      INTEGER, SAVE :: ICNTRL(20)

!     Local variable(s) for multi-processor grid-decomposition
!     information

      INTEGER, SAVE :: STRTCOLMC3, ENDCOLMC3, STRTROWMC3, ENDROWMC3
      INTEGER :: GXOFF, GYOFF

!     Local variable(s) for storing physical quantities. Note that
!     INTERPX does not work if TA, QV, and PRES are declared as
!     DOUBLEPRECISION

      REAL :: TA(NCOLS, NROWS, NLAYS),
     &        QV(NCOLS, NROWS, NLAYS),
     &        PRES(NCOLS, NROWS, NLAYS)

!     External functions

      INTEGER, EXTERNAL :: TIME2SEC, SEC2TIME

!     Interface

      INTERFACE
         SUBROUTINE WR_L2CHK(CGRID, JDATE, JTIME)
            IMPLICIT NONE
            REAL, POINTER, INTENT(IN) :: CGRID(:,:,:,:)
            INTEGER, INTENT(IN)     :: JDATE, JTIME
         END SUBROUTINE WR_L2CHK
      END INTERFACE

C     ------------------------------------------------------------------

!     If no gaseous species are used, then there is nothing to do

      IF (N_GC_SPC .EQ. 0) RETURN

      IF (FIRSTIME) THEN

!     Initialize the log file

         LOGDEV = INIT3()

!     Initialize KPP code

         CALL INITIALIZE()

!     We always integrate between TIN = 0 and TOUT = TSTEP(2)

         TIN = 0.0_dp

!     Set SUN to an unrealistic value so that we hopefuly detect if it
!     gets used by accident

         SUN = 9.9e+50_dp

!     Initialize KPP-CMAQ (SAPRC99) interface

         CALL INIT_CMAQKPP_SAPRC99_INTERFACE()

!     Verbose

         WRITE(LOGDEV, '(A)') ""
         WRITE(LOGDEV, '(A)') "Chemistry module information"
         WRITE(LOGDEV, '(A)') "Generated by KPP for SAPRC99"
         WRITE(LOGDEV, '(A)') ""

!     Display name of species and their order as defined in CMAQ-4.5
!     SAPRC99, with the corresponding indices in KPP.  Use IC
!     temporarily, although here it is not used for a column index

         WRITE(LOGDEV, '(A)') "List of gaseous species:"
         WRITE(LOGDEV, '(A13, 1X, A16, 1X, A12, 1X, A16)')
     &        "Index in CMAQ", "Name in CMAQ", "Index in KPP",
     &        "Name in KPP"
         DO IS = 1, N_GC_SPC
            IC = MAPCMAQ2KPP(IS)
            WRITE(LOGDEV, '(I13, 1X, A16, 1X, I12, 1X, A16)')
     &           IS, GC_SPC(IS), IC, SPC_NAMES(IC)
!     Check that the mapping through MAPKPP2CMAQ works
            IF (MAPKPP2CMAQ(IC) .NE. IS) THEN
               MSG = "Wrong index when doing MAPCMAQ2KPP + MAPKPP2CMAQ"
               CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
            END IF
         END DO

!     Get local (i.e. for the working processor) starting and ending
!     column and row indices

         CALL SUBHFILE(MET_CRO_3D, GXOFF, GYOFF,
     &                 STRTCOLMC3, ENDCOLMC3, STRTROWMC3, ENDROWMC3)

!     Quality check

         IF ((MY_NCOLS .NE. ENDCOLMC3-STRTCOLMC3+1) .OR.
     &        (MY_NROWS .NE. ENDROWMC3-STRTROWMC3+1)) THEN
            MSG = "Confusion about multi-processor H-grid and SUBHFILE"
            CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
         END IF

!     Print information about the chemical mechanism as defined in CMAQ

         CALL CMAQSAPRC99_INFO()

!     Set absolute and relative tolerance

         ATOL(:) = 1.0e-8_dp
         RTOL(:) = 1.0e-3_dp

!     Set variable(s) that control(s) the Rosenbrock solver

         ICNTRL(:) = 0
         ICNTRL(3) = ROS_METHOD

!     Calculate some constants

         CST1 = AVO / RGASUNIV * 1e-12_dp
         CST2 = MWAIR / MWWAT * 1.0e6_dp

!     We are done with initialization

         WRITE(LOGDEV, '(A)') ""
         WRITE(LOGDEV, '(A)') "Done with the initialization of CHEM"
         WRITE(LOGDEV, '(A)') ""

         FIRSTIME = .FALSE.

      END IF ! FIRSTIME

!     Calculate the time step

      TSTEP2_SEC = TIME2SEC(TSTEP(2))
      TOUT = 0.0_dp + TSTEP2_SEC

!     Get the physical data at the middle of the time step

      MIDDATE = JDATE
      MIDTIME = JTIME
      CALL NEXTIME(MIDDATE, MIDTIME, SEC2TIME(TSTEP2_SEC/2))
!     --> air temperature in K
      IF ( .NOT. INTERPX( MET_CRO_3D, "TA              ", PNAME,
     &     STRTCOLMC3,ENDCOLMC3, STRTROWMC3,ENDROWMC3,1,NLAYS,
     &     MIDDATE, MIDTIME, TA ) ) THEN
         MSG = 'Could not read TA from MET_CRO_3D'
         CALL M3EXIT( PNAME, JDATE, JTIME, MSG, XSTAT1 )
      ENDIF
!     ---> water vapor mixing ration in kg H2O / kg air
      IF ( .NOT. INTERPX( MET_CRO_3D, "QV              ", PNAME,
     &     STRTCOLMC3,ENDCOLMC3, STRTROWMC3,ENDROWMC3, 1,NLAYS,
     &     MIDDATE, MIDTIME, QV ) ) THEN
         MSG = 'Could not read QV from MET_CRO_3D'
         CALL M3EXIT( PNAME, JDATE, JTIME, MSG, XSTAT1 )
      ENDIF
!     ---> pressure in Pascals
      IF ( .NOT. INTERPX( MET_CRO_3D, "PRES            ", PNAME,
     &     STRTCOLMC3,ENDCOLMC3, STRTROWMC3,ENDROWMC3, 1,NLAYS,
     &     MIDDATE, MIDTIME, PRES ) ) THEN
         MSG = 'Could not read PRES from MET_CRO_3D'
         CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT1 )
      ENDIF
!     ---> photolysis rates in min-1

      CALL PHOT SUBST_GRID_ID (MIDDATE, MIDTIME, JDATE, JTIME,
     &                         NDARK, RJ)

!     Write the L2CHK checkpoints

      IF (RTO_CHK) CALL WR_L2CHK(CGRID, JDATE, JTIME)

!     Loop over each grid cell for the given processor

      DO IL = 1, NLAYS
         DO IR = 1, MY_NROWS
            DO IC = 1, MY_NCOLS

!     Set temperature, recalculate the CFACTOR (to go from ppmV to
!     molecules per cm-3) and calculate the rate coefficients using
!     KPP's formulae

               TEMP = TA(IC, IR, IL) * 1.0_dp
               CFACTOR = CST1*PRES(IC, IR, IL)/TEMP
               CALL Update_RCONST()

!     Determine if there is sunlight

               LSUNLIGHT = .FALSE.
               DO IPHOT = 1, NPHOTAB
                  IF (RJ(IC, IR, IL, IPHOT) .GT. 0.0) THEN
                     LSUNLIGHT = .TRUE.
                  END IF
               END DO

!     Update the photolysis rates. For now, the mapping of reaction
!     indices between CMAQ and KPP is hard-coded

               IF (.NOT. LSUNLIGHT) THEN
                  RCONST(1)   = 0.0_dp
                  RCONST(15)  = 0.0_dp
                  RCONST(16)  = 0.0_dp
                  RCONST(17)  = 0.0_dp
                  RCONST(18)  = 0.0_dp
                  RCONST(22)  = 0.0_dp
                  RCONST(23)  = 0.0_dp
                  RCONST(28)  = 0.0_dp
                  RCONST(34)  = 0.0_dp
                  RCONST(41)  = 0.0_dp
                  RCONST(123) = 0.0_dp
                  RCONST(124) = 0.0_dp
                  RCONST(131) = 0.0_dp
                  RCONST(134) = 0.0_dp
                  RCONST(137) = 0.0_dp
                  RCONST(139) = 0.0_dp
                  RCONST(142) = 0.0_dp
                  RCONST(144) = 0.0_dp
                  RCONST(145) = 0.0_dp
                  RCONST(146) = 0.0_dp
                  RCONST(149) = 0.0_dp
                  RCONST(152) = 0.0_dp
                  RCONST(159) = 0.0_dp
                  RCONST(165) = 0.0_dp
                  RCONST(169) = 0.0_dp
                  RCONST(173) = 0.0_dp
                  RCONST(175) = 0.0_dp
                  RCONST(177) = 0.0_dp
                  RCONST(181) = 0.0_dp
                  RCONST(183) = 0.0_dp
               ELSE
                  RCONST(1)   = RTDAT(1,   1)*RJ(IC, IR, IL,  1)/60.0_dp
                  RCONST(15)  = RTDAT(1,  15)*RJ(IC, IR, IL,  2)/60.0_dp
                  RCONST(16)  = RTDAT(1,  16)*RJ(IC, IR, IL,  3)/60.0_dp
                  RCONST(17)  = RTDAT(1,  17)*RJ(IC, IR, IL,  4)/60.0_dp
                  RCONST(18)  = RTDAT(1,  18)*RJ(IC, IR, IL,  5)/60.0_dp
                  RCONST(22)  = RTDAT(1,  22)*RJ(IC, IR, IL,  6)/60.0_dp
                  RCONST(23)  = RTDAT(1,  23)*RJ(IC, IR, IL,  7)/60.0_dp
                  RCONST(28)  = RTDAT(1,  28)*RJ(IC, IR, IL,  8)/60.0_dp
                  RCONST(34)  = RTDAT(1,  34)*RJ(IC, IR, IL,  9)/60.0_dp
                  RCONST(41)  = RTDAT(1,  41)*RJ(IC, IR, IL, 10)/60.0_dp
                  RCONST(123) = RTDAT(1, 123)*RJ(IC, IR, IL, 11)/60.0_dp
                  RCONST(124) = RTDAT(1, 124)*RJ(IC, IR, IL, 12)/60.0_dp
                  RCONST(131) = RTDAT(1, 131)*RJ(IC, IR, IL, 13)/60.0_dp
                  RCONST(134) = RTDAT(1, 134)*RJ(IC, IR, IL, 14)/60.0_dp
                  RCONST(137) = RTDAT(1, 137)*RJ(IC, IR, IL, 15)/60.0_dp
                  RCONST(139) = RTDAT(1, 139)*RJ(IC, IR, IL, 16)/60.0_dp
                  RCONST(142) = RTDAT(1, 142)*RJ(IC, IR, IL, 17)/60.0_dp
                  RCONST(144) = RTDAT(1, 144)*RJ(IC, IR, IL, 17)/60.0_dp
                  RCONST(145) = RTDAT(1, 145)*RJ(IC, IR, IL, 18)/60.0_dp
                  RCONST(146) = RTDAT(1, 146)*RJ(IC, IR, IL, 19)/60.0_dp
                  RCONST(149) = RTDAT(1, 149)*RJ(IC, IR, IL, 20)/60.0_dp
                  RCONST(152) = RTDAT(1, 152)*RJ(IC, IR, IL, 21)/60.0_dp
                  RCONST(159) = RTDAT(1, 159)*RJ(IC, IR, IL, 22)/60.0_dp
                  RCONST(165) = RTDAT(1, 165)*RJ(IC, IR, IL, 23)/60.0_dp
                  RCONST(169) = RTDAT(1, 169)*RJ(IC, IR, IL, 23)/60.0_dp
                  RCONST(173) = RTDAT(1, 173)*RJ(IC, IR, IL, 23)/60.0_dp
                  RCONST(175) = RTDAT(1, 175)*RJ(IC, IR, IL, 16)/60.0_dp
                  RCONST(177) = RTDAT(1, 177)*RJ(IC, IR, IL, 24)/60.0_dp
                  RCONST(181) = RTDAT(1, 181)*RJ(IC, IR, IL, 25)/60.0_dp
                  RCONST(183) = RTDAT(1, 183)*RJ(IC, IR, IL, 23)/60.0_dp
               END IF

!     Load the values of the concentrations of CMAQ (SAPRC99) species at
!     the beginning of the timestep. They are converted from ppmV to
!     molecules per cm-3

               DO IS = 1, N_GC_SPC
                  C(MAPCMAQ2KPP(IS)) = MAX(
     &                 CGRID(IC, IR, IL, IS)*CFACTOR,
     &                 MIN_C_INTEGRATE)
               END DO

!     Load the concentrations of the other species (species considered
!     to be non-reactive or having constant mole fractions in the
!     KPP-generated code). Value of x_O2, x_H2, and X_CH4 are from
!     .../CCTM/chem/ebi_saprc99/hrcalcks.F

               C(I_KPP_XC)  = 0.0_dp
               C(I_KPP_XN)  = 0.0_dp
               C(I_KPP_AIR) = 1.0e+6_dp * CFACTOR
               C(I_KPP_O2)  = 0.2095e+6_dp * CFACTOR
               C(I_KPP_H2O) = MAX(QV(IC, IR, IL)*CST2*CFACTOR, 0.0_dp)
               C(I_KPP_H2)  = 0.56_dp * CFACTOR
               C(I_KPP_CH4) = 1.85_dp * CFACTOR

!     Call the integrator and check on its exit status

               CALL INTEGRATE(TIN, TOUT, ICNTRL_U=ICNTRL, IERR_U=STATUS)
               IF (STATUS .NE. 1) THEN
                  MSG = "INTEGRATE returned an error"
                  CALL M3EXIT(PNAME, JDATE, JTIME, MSG, XSTAT2)
               END IF

!     Update concentration array

               DO IS = 1, N_GC_SPC
                  CGRID(IC, IR, IL, IS) =
     &                 SNGL(C(MAPCMAQ2KPP(IS))/CFACTOR)
               END DO

            END DO
         END DO
      END DO

      RETURN

      END SUBROUTINE CHEM
