
C***********************************************************************
C   Portions of Models-3/CMAQ software were developed or based on      *
C   information from various groups: Federal Government employees,     *
C   contractors working on a United States Government contract, and    *
C   non-Federal sources (including research institutions).  These      *
C   research institutions have given the Government permission to      *
C   use, prepare derivative works, and distribute copies of their      *
C   work in Models-3/CMAQ to the public and to permit others to do     *
C   so.  EPA therefore grants similar permissions for use of the       *
C   Models-3/CMAQ software, but users are requested to provide copies  *
C   of derivative works to the Government without restrictions as to   *
C   use by others.  Users are responsible for acquiring their own      *
C   copies of commercial software associated with Models-3/CMAQ and    *
C   for complying with vendor requirements.  Software copyrights by    *
C   the MCNC Environmental Modeling Center are used with their         *
C   permissions subject to the above restrictions.                     *
C***********************************************************************

      MODULE CMAQKPP_SAPRC99_INTERFACE

C     BY: Lucas A. J. Bastien. Parts of this code (or all of it) may be
C     based on or inspired by pre-existing CMAQ and/or CMAQ adjoint
C     code, with possibly some sections copied without modification.
C
C     This module contains utilities to integrate the code generated by
C     KPP for the saprc99 mechanism with CMAQ v4.5-ADJ.

      USE saprc99ros_Precision

      IMPLICIT NONE

!     Some species in the KPP list are not in the CMAQ list. They are
!     the species considered to be non-reactive or having constant
!     concentrations in the KPP-generated code. We have XC (lost carbon
!     or carbon in unreactive products), XN (lost nitrogen or nitrogen
!     in unreactive products), AIR, O2, H2O, H2, CH4. The following
!     variables are the indices of these species in the KPP list of
!     species

      INTEGER, SAVE :: I_KPP_XC
      INTEGER, SAVE :: I_KPP_XN
      INTEGER, SAVE :: I_KPP_AIR
      INTEGER, SAVE :: I_KPP_O2
      INTEGER, SAVE :: I_KPP_H2O
      INTEGER, SAVE :: I_KPP_H2
      INTEGER, SAVE :: I_KPP_CH4

!     Maps to go between species lists of CMAQ (SAPRC99) and
!     KPP-generated code

      INTEGER, ALLOCATABLE, SAVE :: MAPCMAQ2KPP(:)
      INTEGER, ALLOCATABLE, SAVE :: MAPKPP2CMAQ(:)

!     Minumum concentration value passed to the KPP integrator. This
!     value was found in the KPP driver provided in the default CMAQ
!     v4.5-ADJ code for the cb4 mechanism

      REAL(KIND=dp), SAVE :: MIN_C_INTEGRATE = 0.0_dp + 1.0D-30

!     Selection of a partcicular Rosenbrock method for the chemistry
!     solver (Read from environmnent at run-time)
!     0: Rodas3 (default)
!     1: Ros2
!     2: Ros3
!     3: Ros4
!     4: Rodas3
!     5: Rodas4
      INTEGER, SAVE :: ROS_METHOD

!     ==================================================================

      CONTAINS

!     ==================================================================

      INTEGER FUNCTION ISPC_KPP(SPC)

      ! Returns index in KPP list of species named SPC. Be sure to pass
      ! a 16-character string (not shorter) as its argument SPC

!     Module(s)

      USE saprc99ros_Parameters
      USE saprc99ros_Monitor

      IMPLICIT NONE

!     Dummy variable(s)

      CHARACTER(16), INTENT(IN) :: SPC

!     Include file(s)

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions

!     Local variable(s)

      CHARACTER(16) :: PNAME = "ISPC_KPP"
      CHARACTER(96) :: MSG = ""
      LOGICAL :: STILL_LOOKING
      INTEGER :: I

!     Look at the table of species names as defined in KPP for the name
!     of the species

      I = 1
      STILL_LOOKING = .TRUE.
      DO WHILE ((STILL_LOOKING) .AND. (I .LE. NSPEC))
         IF (TRIM(SPC_NAMES(I)) .EQ. TRIM(SPC)) THEN
            STILL_LOOKING = .FALSE.
         ELSE
            I = I + 1
         END IF
      END DO

!     Error if we did not find the species

      IF (STILL_LOOKING) THEN
         MSG = "Species (" // TRIM(SPC) //
     &         ") not found in KPP list"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      ELSE
         ISPC_KPP = I
      END IF

!     Check the range of the returned index

      IF ((ISPC_KPP .LT. 1) .OR. (ISPC_KPP) .GT. NSPEC) THEN
         MSG = "KPP index out of range"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

      END FUNCTION ISPC_KPP

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      INTEGER FUNCTION ISPC_CMAQ2KPP(ISPC)

!     Returns the index in KPP list of species having index ISPC in CMAQ
!     list

!     Module(s)

      USE saprc99ros_Parameters
      USE saprc99ros_Monitor

      IMPLICIT NONE

!     Dummy variable(s)

      INTEGER, INTENT(IN) :: ISPC

!     Include file(s)

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions
      INCLUDE SUBST_GC_SPC  ! Gas phase species information

!     Local variable(s)

      CHARACTER(16) :: PNAME = "ISPC_CMAQ2KPP"
      CHARACTER(96) :: MSG = ""
      CHARACTER(16) :: SPC

!     ISP should be an acceptable index in CAMQ

      IF (ISPC .GT. N_GC_SPC) THEN
         MSG = "ISPC too big in ISPC_CMAQ2KPP"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

!     Get the name of the species that we are talking about. A few
!     species have a different name between CMAQ SAPRC99 files and
!     KPP-generated code. We correct that manually here

      SPC = TRIM(GC_SPC(ISPC))

      IF (SPC .EQ. "O1D2") THEN
         SPC = "O1D             "
      ELSE IF (SPC .EQ. "HO") THEN
         SPC = "OH              "
      ELSE IF (SPC .EQ. "HO2H") THEN
         SPC = "H2O2            "
      ELSE IF (SPC .EQ. "SULF") THEN
         SPC = "H2SO4           "
      ELSEIF (SPC .EQ. "TRP1") THEN
         SPC = "TERP            "
      END IF

      ISPC_CMAQ2KPP = ISPC_KPP(SPC)

!     Since the species is defined in CMAQ (SAPRC99), the returned index
!     should correspond to a variable species in KPP

      IF ((ISPC_CMAQ2KPP .LT. NVARST) .OR.
     &     (ISPC_CMAQ2KPP .GE. NVARST+NVAR)) THEN
         MSG = "KPP index for CMAQ species out of range"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

      END FUNCTION ISPC_CMAQ2KPP

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      SUBROUTINE INIT_CMAQKPP_SAPRC99_INTERFACE()

!     Initialize the interface between KPP-generated code and CMAQ

!     Module(s)

      USE saprc99ros_Parameters

      IMPLICIT NONE

!     Include file(s)

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions
      INCLUDE SUBST_IODECL  ! I/O definitions and declarations
      INCLUDE SUBST_GC_SPC  ! Gas phase species information

!     Local variable(s)

      CHARACTER(16) :: PNAME = "INIT_CMAQKPP..."
      CHARACTER(96) :: MSG = ""
      LOGICAL, SAVE ::FIRSTIME = .TRUE.
      INTEGER, SAVE :: LOGDEV
      CHARACTER(16) :: SPC
      INTEGER :: I, J
      INTEGER :: STATUS
      CHARACTER(80) :: VARDESC

!     External function(s)

      INTEGER, EXTERNAL :: ENVINT

      IF (FIRSTIME) THEN

         LOGDEV = INIT3()

!     As of now, the KPP-code integrated in CMAQ assumes that the
!     species list in the KPP-generated code is longer than the species
!     list in CMAQ (SAPRC99)

         IF (N_GC_SPC .GT. NSPEC) THEN
            MSG = "Expecting more species in KPP than in CMAQ"
            CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
         END IF

!     Get indices in KPP-generated code table of constant-concentration
!     or non-reactive products. Check that XC and XN are considered as
!     variable species in KPP and that the others are considered as
!     fixed species

         I_KPP_XC  = ISPC_KPP("XC              ")
         I_KPP_XN  = ISPC_KPP("XN              ")
         I_KPP_AIR = ISPC_KPP("AIR             ")
         I_KPP_O2  = ISPC_KPP("O2              ")
         I_KPP_H2O = ISPC_KPP("H2O             ")
         I_KPP_H2  = ISPC_KPP("H2              ")
         I_KPP_CH4 = ISPC_KPP("CH4             ")

         IF ((I_KPP_XC .LT. NVARST) .OR.
     &       (I_KPP_XC .GE. NVARST+NVAR) .OR.
     &       (I_KPP_XN .LT. NVARST) .OR.
     &       (I_KPP_XN .GE. NVARST+NVAR) .OR.
     &       (I_KPP_AIR .LT. NFIXST) .OR.
     &       (I_KPP_AIR .GE. NFIXST+NFIX) .OR.
     &       (I_KPP_O2 .LT. NFIXST) .OR.
     &       (I_KPP_O2 .GE. NFIXST+NFIX) .OR.
     &       (I_KPP_H2O .LT. NFIXST) .OR.
     &       (I_KPP_H2O .GE. NFIXST+NFIX) .OR.
     &       (I_KPP_H2 .LT. NFIXST) .OR.
     &       (I_KPP_H2 .GE. NFIXST+NFIX) .OR.
     &       (I_KPP_CH4 .LT. NFIXST) .OR.
     &       (I_KPP_CH4 .GE. NFIXST+NFIX)) THEN
            MSG = "Index out of range for a KPP special species"
            CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
         END IF

!     Allocate map tables and initialize the values to -1

         ALLOCATE(MAPCMAQ2KPP(N_GC_SPC), STAT=STATUS)
         IF (STATUS .NE. 0) THEN
            MSG = "MAPCMAQ2KPP memory allocation failed"
            CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
         END IF
         MAPCMAQ2KPP = -1

         ALLOCATE (MAPKPP2CMAQ(NSPEC), STAT=STATUS)
         IF (STATUS .NE. 0) THEN
            MSG = "MAPKPP2CMAQ memory allocation failed"
            CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
         END IF
         MAPKPP2CMAQ = -1

!     Fill values in map tables. The index returned by ISPC_CMAQ2KPP
!     should correspond to a variable species in KPP

         DO I = 1, N_GC_SPC
            J = ISPC_CMAQ2KPP(I)
            MAPCMAQ2KPP(I) = J
            MAPKPP2CMAQ(J) = I
         END DO

!     Read from the environment what Rosenbrock method should be used

         VARDESC = "Choice of Rosenbrock method for chemistry"
         ROS_METHOD = ENVINT( "ROS_METHOD", VARDESC, 0, STATUS )
         IF ( STATUS .NE. 0 ) THEN
            WRITE( LOGDEV, '(5X, A)' ) VARDESC
            IF ( STATUS .EQ. 1 ) THEN
               MSG = "Environment variable improperly formatted"
               CALL M3EXIT( PNAME, 0, 0, MSG, XSTAT2 )
            ELSE IF ( STATUS .EQ. -1 ) THEN
               MSG = "Environment variable set but empty Using default:"
               WRITE( LOGDEV, '(5X, A, I6)' ) MSG, 0
            ELSE IF ( STATUS .EQ. -2 ) THEN
               MSG = "Environment variable not set ... Using default:"
               WRITE( LOGDEV, '(5X, A, I6)' ) MSG, 0
            END IF
         END IF

         FIRSTIME = .FALSE.

      ELSE

         MSG = "INIT_CMAQKPP_SAPRC99_INTERFACE has already been called"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)

      END IF

      RETURN

      END SUBROUTINE INIT_CMAQKPP_SAPRC99_INTERFACE

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      SUBROUTINE WRITE_REACTION_EXP(IR, UNIT)

!     Writes the expression of reaction number IT in the CMAQ list in
!     file with logical unit number UNITS

      IMPLICIT NONE

!     Dummy variable(s)

      INTEGER, INTENT(IN) :: IR, UNIT

!     Include file(s)

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions
      INCLUDE SUBST_GC_SPC  ! Gas phase species information
      INCLUDE SUBST_RXCMMN  ! Chemical mechanism information header

!     Local variable(s)

      CHARACTER(16) :: PNAME = "WRITE_REAC.."
      CHARACTER(96) :: MSG = ""
      INTEGER :: I
      LOGICAL :: PLUS_SIGN

!     Check range of input

      IF ((IR .LT. 1) .OR. (IR .GT. NRXNS)) THEN
         MSG = "Index of reaction out of range"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

!     Reactants

      PLUS_SIGN = .FALSE.
      DO I = 1, 3
         IF (IRR(IR, I) .GT. 0) THEN
            IF (PLUS_SIGN) THEN
               WRITE(UNIT, '(5A)', ADVANCE="NO") " + "
            ELSE
               WRITE(UNIT, '(A)', ADVANCE="NO") "     "
               PLUS_SIGN = .TRUE.
            END IF
            WRITE(UNIT, '(A)', ADVANCE="NO") TRIM(GC_SPC(IRR(IR, I)))
         END IF
      END DO

!     Equal sign

      WRITE(UNIT, '(A)', ADVANCE="NO") " = "

!     Products

      PLUS_SIGN = .FALSE.
      DO I = 4, MXPRD + 3
         IF (IRR(IR, I) .GT. 0) THEN
            IF (PLUS_SIGN) THEN
               WRITE(UNIT, '(A)', ADVANCE="NO") " + "
            ELSE
               PLUS_SIGN = .TRUE.
            END IF
            WRITE(UNIT, '(A)', ADVANCE="NO") TRIM(GC_SPC(IRR(IR, I)))
         END IF
      END DO

!     Add end of line

      WRITE(UNIT, '(A)') ""

      END SUBROUTINE WRITE_REACTION_EXP

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      SUBROUTINE WRITE_REACTION_RATE(IR, UNIT)

!     Writes the expression of reaction rate for reaction number IT in
!     the CMAQ list in file with logical unit number UNITS. As of now,
!     only photolysis reactions are supported

      IMPLICIT NONE

!     Dummy variable(s)

      INTEGER, INTENT(IN) :: IR, UNIT

!     Include file(s)

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions
      INCLUDE SUBST_RXCMMN  ! Chemical mechanism information header

!     Local variable(s)

      CHARACTER(16) :: PNAME = "WRITE_REAC.."
      CHARACTER(96) :: MSG = ""
      INTEGER :: RTYPE

!     Check range of input

      IF ((IR .LT. 1) .OR. (IR .GT. NRXNS)) THEN
         MSG = "Index of reaction out of range"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

!     Format of reaction rate constant depends on the type of
!     reaction. Information on format of rate constant is taken from
!     HRCALCKS
!     Type 0: Photolysis
!     Type 1: k=A
!     Type 2: k=A*(T/300)**B
!     Type 3: k=A*exp(C/T)
!     Type 4: k=A*((T/300)**B)*exp(C/T)
!     Type 5: k=multipliers of other reactions
!     Type 6: k=multipliers of other reactions
!     Type 7: k=A*(1+0.6*P)
!     Type 8: k=fall offs and special type %2

      WRITE(UNIT, '(A)', ADVANCE="NO") "     k = "
      RTYPE = KTYPE(IR)
      IF (RTYPE .EQ. 0) THEN
         WRITE(UNIT, '(E12.6)', ADVANCE="NO") RTDAT(1, IR)
      ELSE
         MSG = "Unrecognized or non-implemented reaction type"
         CALL M3EXIT(PNAME, 0, 0, MSG, XSTAT2)
      END IF

!     Add end of line

      WRITE(UNIT, '(A)') ""

      END SUBROUTINE WRITE_REACTION_RATE

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      SUBROUTINE CMAQSAPRC99_INFO()

!     Write information about CMAQ SAPRC99 into the logfile

      IMPLICIT NONE

!     Include file(s)

      INCLUDE SUBST_IODECL ! I/O definitions and declarations
      INCLUDE SUBST_RXCMMN ! Chemical mechanism information header

!     Local variable(s)

      LOGICAL, SAVE ::FIRSTIME = .TRUE.
      INTEGER, SAVE :: LOGDEV
      INTEGER :: I, J
      INTEGER :: IR  ! Index of the reaction
      INTEGER :: IR2 ! Index of the reaction in RJIN

      IF (FIRSTIME) THEN

         LOGDEV = INIT3()

         FIRSTIME = .FALSE.

      END IF

!     Photolysis reactions

      WRITE(LOGDEV, '(A)') "Photolysis reactions"
      DO I = 1, NMPHOT
         IR = IPH(I, 1)
         IR2 = IPH(I, 2)
         WRITE(LOGDEV, '(I3, A, I3, A, I3, A, I3, A, L1, 3X)')
     &        I, ". Reaction ", IR, "/", IR2, ". Type ", KTYPE(IR),
     &        ", Absolute: ", IPH(I, 3) .NE. 0
         CALL WRITE_REACTION_EXP(IR, LOGDEV)
         CALL WRITE_REACTION_RATE(IR, LOGDEV)
         WRITE(LOGDEV, '(A)') ""
      END DO

      END SUBROUTINE CMAQSAPRC99_INFO

!     -------------------------------------------------------------------------
!     -------------------------------------------------------------------------

      END MODULE CMAQKPP_SAPRC99_INTERFACE
