
C***********************************************************************
C   Portions of Models-3/CMAQ software were developed or based on      *
C   information from various groups: Federal Government employees,     *
C   contractors working on a United States Government contract, and    *
C   non-Federal sources (including research institutions).  These      *
C   research institutions have given the Government permission to      *
C   use, prepare derivative works, and distribute copies of their      *
C   work in Models-3/CMAQ to the public and to permit others to do     *
C   so.  EPA therefore grants similar permissions for use of the       *
C   Models-3/CMAQ software, but users are requested to provide copies  *
C   of derivative works to the Government without restrictions as to   *
C   use by others.  Users are responsible for acquiring their own      *
C   copies of commercial software associated with Models-3/CMAQ and    *
C   for complying with vendor requirements.  Software copyrights by    *
C   the MCNC Environmental Modeling Center are used with their         *
C   permissions subject to the above restrictions.                     *
C***********************************************************************

      SUBROUTINE ADVSTEPMIN(NLAYS, NSTEPS, STDATE, STTIME, TSTEP,
     &                      ASTEP, NREPS)

C     BY: Lucas A. J. Bastien. Parts of this code (or all of it) may be
C     based on or inspired by pre-existing CMAQ and/or CMAQ adjoint
C     code, with possibly some sections copied without modification.
C
C     PURPOSE (SUMMARY): This subroutine calculates the minimum
C     advection and synchronization time steps throughout the
C     simulation. It also calculates the corresponding number of
C     synchronization time steps per output time step.
C
C     PURPOSE (DETAILED): In CMAQ-v4.5, advection and synchronization
C     time steps are re-calculated at each output time step and depend
C     on the meteorology for that time step (the calculation revolves
C     around respecting the CFL criterion). Advection and
C     synchronization time steps can be different between one output
C     time step and the next one. Note that for each output time step,
C     one advection time step is calculated for each vertical layer. In
C     CMAQ-v4.5-ADJ, the concentration field is checkpointed at each
C     synchronization time step. Therefore the synchronization time step
C     needs to be constant thoughout the simulation. This subroutine
C     goes through all the output time steps of the simulation,
C     calculates the advection and synchronization time steps for each
C     of the output time steps, and returns the minimum of those
C     advection and synchronization time steps. This subroutine also
C     returns the corresponding number of synchronization time steps per
C     output time step. This subroutine should be called before creating
C     the checkpoint files (which will use the minimum synchronization
C     time step as time step) and before the forward and backward
C     loops. This subroutine checks that at each new output time step,
C     either all new advection times steps are smaller than the old ones
C     or all new advection times steps are larger than the old ones (the
C     comparison is done on a per layer basis -- remember that one
C     advection time step is calculated for each layer). If, among the
C     newly calculated advecion time steps, some are smaller than the
C     old ones and some are larger than the old one (on a per layer
C     basis), then the question "are new time steps smaller than the old
C     ones" cannot be answered by "yes" or "no". In that case, the
C     subroutine exits with error. To avoid this issue, the user can
C     reduce the value of the environment variable SIGMA_SYNC_TOP at run
C     time.

C     ------------------------------------------------------------------

      IMPLICIT NONE

C     Include files

      INCLUDE SUBST_IOPARMS ! I/O parameters definitions

C     Dummy arguments

      INTEGER, INTENT(IN) :: NSTEPS        ! Number of output time steps
      INTEGER, INTENT(IN) :: NLAYS         ! Number of vertical layers
      INTEGER, INTENT(IN) :: STDATE        ! Start date, format YYYYDDD
      INTEGER, INTENT(IN) :: STTIME        ! Start time, format HHMMSS
      INTEGER, INTENT(INOUT) :: TSTEP(2)   ! Output time step: TSTEP(1);
                                           ! and synchronization time
                                           ! step: TSTEP(2)
      INTEGER, INTENT(OUT) :: ASTEP(NLAYS) ! Advection time step for
                                           ! each layer
      INTEGER, INTENT(OUT) :: NREPS        ! Number of synchronization
                                           ! time steps per output time
                                           ! steps

C     Local variables

      CHARACTER(16) :: PNAME  ! Subroutine name
      INTEGER       :: STATUS ! Return status
      CHARACTER(70) :: MSG    ! Status/warning/error message
      INTEGER :: TMP_TSTEP(2) ! Temporary place holder for newly
                              ! calculated TSTEP
      INTEGER, ALLOCATABLE :: TMP_ASTEP(:) ! Temporary place holder for
                                           ! newly calculated ASTEP
      INTEGER :: TMP_NREPS    ! Temporary place holder for newly
                              ! calculated NREPS
      INTEGER :: JDATE        ! Current date, format YYYYDDD
      INTEGER :: JTIME        ! Current time, format HHMMSS
      INTEGER :: ISTEP        ! Variable to iterate over output time
                              ! steps
      INTEGER :: J            ! Iteration variable
      LOGICAL :: NORMAL       ! Is everything as expected?
      LOGICAL :: ISONEBIGGER  ! Compare old and new advection time step
      LOGICAL :: ISONESMALLER ! Compare old and new advection time step

C     Interface

      INTERFACE
         SUBROUTINE ADVSTEP ( JDATE, JTIME, TSTEP, ASTEP, NREPS )
            IMPLICIT NONE
            INTEGER, INTENT(IN)  :: JDATE, JTIME
            INTEGER, INTENT(IN)  :: TSTEP(2)
            INTEGER, INTENT(IN)  :: ASTEP(:)
            INTEGER, INTENT(OUT) :: NREPS
         END SUBROUTINE ADVSTEP
      END INTERFACE

C     ------------------------------------------------------------------

      PNAME = "ADVSTEPMIN"

C     Allocate the temporary buffer for ASTEP

      ALLOCATE ( TMP_ASTEP( NLAYS ), STAT = STATUS )
      IF ( STATUS .NE. 0 ) THEN
         MSG = 'TMP_ASTEP memory allocation failed'
         CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT2 )
      END IF

C     Initialize

      JDATE = STDATE
      JTIME = STTIME
      TMP_TSTEP(1) = TSTEP(1)
      TSTEP(2) = 999999
      ASTEP = 999999
      NREPS = -1

C     Calculate sync and adv time steps for each time step

      DO ISTEP = 1, NSTEPS

         CALL ADVSTEP ( JDATE, JTIME, TMP_TSTEP, TMP_ASTEP, TMP_NREPS )

C     Check that TSTEP(1) has not changed

         IF ( TMP_TSTEP(1) .NE. TSTEP(1) ) THEN
            MSG = "TSTEP(1) has changed"
            CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT2 )
         END IF

C     Either all new advection time steps are smaller than or equal to
C     the old ones, or all new advection time steps are larger than or
C     equal to the old ones. If it is not the case, exit with error

         NORMAL = .TRUE.
         ISONESMALLER = .FALSE.
         ISONEBIGGER = .FALSE.

         J = 1
         DO WHILE ( (J .LE. NLAYS) .AND. NORMAL )
            IF ( TMP_ASTEP(J) .LT. ASTEP(J) ) ISONESMALLER = .TRUE.
            IF ( TMP_ASTEP(J) .GT. ASTEP(J) ) ISONEBIGGER = .TRUE.
            NORMAL = ( .NOT. (ISONESMALLER .AND. ISONEBIGGER) )
            J = J+1
         END DO

         IF (TMP_TSTEP(2) .LT. TSTEP(2)) THEN
            NORMAL = NORMAL .AND. ( .NOT. ISONEBIGGER ) .AND.
     &                            (TMP_NREPS .GT. NREPS)
         END IF

         IF (TMP_TSTEP(2) .GT. TSTEP(2)) THEN
            NORMAL = NORMAL .AND. ( .NOT. ISONESMALLER ) .AND.
     &                            (TMP_NREPS .LT. NREPS)
         END IF

         IF (TMP_TSTEP(2) .EQ. TSTEP(2)) THEN
            NORMAL = NORMAL .AND. (TMP_NREPS .EQ. NREPS)
         END IF

         IF ( .NOT. NORMAL ) THEN
            MSG = "Something wrong. Maybe try lowering SIGMA_SYNC_TOP?"
            CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT2 )
         END IF

C     Record new time step if necessary

         IF ( ISONESMALLER ) THEN
            TSTEP(2) = TMP_TSTEP(2)
            DO J=1,NLAYS
               ASTEP(J) = TMP_ASTEP(J)
            END DO
            NREPS = TMP_NREPS
         END IF

C     Advance time step

         CALL NEXTIME( JDATE, JTIME, TSTEP(1) )

      END DO

C     Deallocate TMP_ASTEP

      DEALLOCATE ( TMP_ASTEP, STAT = STATUS )
      IF ( STATUS .NE. 0 ) THEN
         MSG = 'TMP_ASTEP memory deallocation failed'
         CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT2 )
      END IF

C     Check that TSTEP(1) has not changed

      IF (TMP_TSTEP(1) .NE. TSTEP(1)) THEN
         MSG = "TSTEP(1) has been changed"
         CALL M3EXIT ( PNAME, JDATE, JTIME, MSG, XSTAT2 )
      END IF

      RETURN

      END SUBROUTINE ADVSTEPMIN
