# $Id: simul_parameters.R,v 1.5 2017/08/16 15:09:49 stefan Exp stefan $
 
# Supplement of: 
# Gas-Chromatography using ice coated fused silica-columns: Study of adsorption of sulfur dioxide on water-ice
#
# Set of simulation parameters and utility functions

# Install required packages
CheckInstalledPackage <- function(package) {
  if (!package %in% installed.packages()) install.packages(package)
}
CheckInstalledPackage("ReacTran")
library(ReacTran)

# Simulation parameters
parameters <- c(
	R = 8.3144598,										# Gas constant
	zeta = (4.849e-11*247 - 9.345e-09),			# Parameter of Temkin isotherm
	Khenry = 1.105e-8,								# Henry's adsorption constant
	Ktemkin = 1e-5,       							# Temkin's adsorption constant
	Nz = 100,       									# Number off cells in axial direction
   Ny = 30, 											# Number off cells in radial direction
	points = 400,      								# Number of points calculated for chromatogram
	tmax = 50,											# Time range
	Radius = 265e-6,									# Column radius
	T = 247,												# Column temperature
	Tr = 295.0,											# Room Temperature
	p0 = 1.013e5,										# Atmospheric pressure
	MassFlow = 1.52e-5,	    						# Carrier gas mass flow
	l = 8.07,											# Length of ice coated part of the column
	sigma = 0.5,										# Width of injected peak
	voidTime = 6.04,									# Column void time
	xSO2 = 25,              						# SO2 mixing ratio [ppm]
	Da = 3e-4,        								# Axial dispersion coefficient
	Dy = 2e-13,    									# solid phase diffusion coefficient
	h = 2e-6, 											# Eisfildicke 
	H = 50,												# dimensionless Henry coefficient
	peak = TRUE											# TRUE = injektion of peaks, FALSE = breaktrough curves
)

# Graphics parameters
width <- 3.5
height <- 3.5
mar <- c(5.1, 4.1, 0.2, 0.2)
mai <- c(1.02, 0.82, 0.04, 0.04) 
family <-"Helvetica"
pointsize <-8
lwd <- 1

p <- as.list(parameters)

# Boundary condition: injection of SO2 as peak
CGauss <- function (x,sigma) {
	x <- x/sigma                   
  	x <- x*x;
   return <- 1/sqrt(2*pi)/sigma*exp(-0.5*x);
}

# Boundary condition: injection of SO2 as step impulse using
# the comulative distibution function, 
# https://en.wikipedia.org/wiki/Normal_distribution#Cumulative_distribution_function
Phi <- function(x,sigma) {
	return <- pnorm(x/sigma)
}	

# Capacity ratio (Temkin isotherm)
kapazv <- function(y,parameters) {
	with(as.list(parameters), {
		return <- 2*R*T/Radius*(zeta/(zeta/Ktemkin + 1e-6*y*p0) + Khenry)		
	})
}

# Inverse capacity ratio
invkapaz <- function(tnetto,parameters) {
		with(as.list(parameters), {
		return <- 1e6/p0*zeta*(1/(Radius*tnetto/(2*R*T*voidTime)-Khenry) - 1/Ktemkin)		
	})
}
