# $Id: 2Dmodel.R,v 1.10 2017/08/16 15:10:21 stefan Exp stefan $
# 
# Supplement of: 
# Gas-Chromatography using ice coated fused silica-columns: Study of adsorption of sulfur dioxide on water-ice
#
# Simulation of adsorption on and absorption into the solid phase

source("simul_parameters.R")

fakkapa <- function(y,parameters) {
	h <- kapazv(y,parameters)
	return <- 1/(1 + h)
}

peak.model <- function(time, y, parms) {
	with(as.list(parms), {
	
		CONC  <- matrix(nrow=Nz, ncol=Ny, data = y)
      KF <- matrix(nrow=Nz, ncol=Ny, data = 1.0)		
		cg <- CONC[,1]
		KF[,1] <- fakkapa(cg,parms)
		v <- l / voidTime
		v.grid$x.mid[,1] <- v
		v.grid$x.int[,1] <- c(v, v.grid$x.mid[,1]) 
      flux.x.up <- array(dim=Ny, data=0)
      flux.x.up[[1]] <- v*xSO2*ifelse(peak==TRUE,CGauss(time - voidTime,sigma),Phi(time - voidTime,sigma))
		transport <- tran.2D(CONC, 
			flux.x.up=flux.x.up,
			flux.y.up=0,
			flux.y.down=0,
			grid=grid2D,
			AFDW.x =1,
			AFDW.y=0.5,
			D.grid=D.grid,
			v.grid=v.grid,
			A.grid = A.grid
		)
		dCONC <- KF*transport$dC	
		return <- list(dCONC, xSO2out=CONC[Nz,1])
	})
}

# 2D-Simulation
simul <- function(parameters)
{
	p <- as.list(parameters)

	x.grid <<- setup.grid.1D(L=p$l, N=p$Nz)
	y.grid <<- setup.grid.1D(x.down=c(p$Radius/2,
	p$Radius/2+p$H*p$h),
	L=c(p$Radius/2,p$H*p$h),
	N=c(1,p$Ny-1))
	grid2D <<- setup.grid.2D(x.grid, y.grid)
	D.grid <<- setup.prop.2D(value=p$Da, y.value = p$Dy*p$H*p$H, grid = grid2D)
	D.grid$y.int[,1] <<- 0
	D.grid$y.int[,2] <<- 0.5*p$Dy*p$H*p$H*(p$Radius*(p$Ny-1)/(2*p$h*p$H) + 1)
	D.grid$y.int[,p$Ny+1] <<- 0
	A.grid <<- setup.prop.2D(value=pi*p$Radius*p$Radius, y.value=2*pi*p$Radius*p$l/p$Nz, grid=grid2D)
	v.grid <<- setup.prop.2D(value= 0, grid = grid2D)
	times <- seq(0, p$tmax, by=(p$tmax/p$points))
	xSO2ini <- matrix(nrow = p$Nz, ncol = p$Ny, data = 0)
	out <- ode.2D(func=peak.model, y=xSO2ini, parms=parameters, times=times, dim=c(p$Ny,p$Nz),lrw = 320000)
	tnetto <- times - 2 * p$voidTime	
	return <- data.frame(x=tnetto,y=out)
}

pdf(file="2Dmodel.pdf",width=width,height=height,family=family,pointsize=pointsize)
par(mai=mai,mar=mar)
parameters["xSO2"] <- 25
dfout <- simul(parameters)
plot(dfout$x,dfout$y.xSO2out, type="l",xlim=c(-3,40),
  xlab="adjusted retention time / [s]", 
  ylab=expression(paste(italic(x), "(", SO[2], ") / ppm")),lty=1,col="black")
  
parameters["xSO2"] <- 20
dfout <- simul(parameters)
lines(dfout$x,dfout$y.xSO2out, type="l",new=FALSE,col="black",lty=1)

parameters["xSO2"] <- 15
dfout <- simul(parameters)
lines(dfout$x,dfout$y.xSO2out, type="l",new=FALSE,col="black",lty=1)

parameters["xSO2"] <- 10
dfout <- simul(parameters)
lines(dfout$x,dfout$y.xSO2out, type="l",new=FALSE,col="black",lty=1)

parameters["xSO2"] <- 7
dfout <- simul(parameters)
lines(dfout$x,dfout$y.xSO2out, type="l",new=FALSE,col="black",lty=1)

parameters["xSO2"] <- 5
dfout <- simul(parameters)
lines(dfout$x,dfout$y.xSO2out, type="l",new=FALSE,col="black",lty=1)

# position of peak maxima
t <- seq(p$tmax/40, p$tmax, by=(p$tmax/40))
lines(t,invkapaz(t,parameters),lty=3,col="blue",lwd=2)
dev.off()
