# -*- coding: utf-8 -*-
"""
Modeled snowflake (radar) backscatter cross sections.

@author: mgergely
"""

import numpy as np
import matplotlib.pyplot as plt
        
        
####################################
#%% Data


# Data directory
indir = './data/'

# Load snowflake (single) mass-equivalent ice sphere radii req [mm]
req = np.loadtxt(indir + 'req_mm.txt')
# Load snowflake diameterts D [mm] corresponding to req
D = np.loadtxt(indir + 'D_mm.txt')


# Load backscatter cross sections sigma_b(D; xi) [mm2] for collections of randomly distributed ice spheres determined according to Sect. 3.3 (for Ncl = xi^3 = 1, 4, 8, 16, 27, 64, 125, 216), at 10, 14, 35, 94 GHz
sigmab_10 = np.loadtxt(indir + 'sigmab_mm2_10GHz.txt')
sigmab_14 = np.loadtxt(indir + 'sigmab_mm2_14GHz.txt')
sigmab_35 = np.loadtxt(indir + 'sigmab_mm2_35GHz.txt')
sigmab_94 = np.loadtxt(indir + 'sigmab_mm2_94GHz.txt')


###################################
#%% Plot for 35 GHz


# Minimum and maximum of sigma_b range up to Ncl=125 or xi=5
sigmab_35_min = np.amin(sigmab_35[:-1,:], axis=0)
sigmab_35_max = np.amax(sigmab_35[:-1,:], axis=0)


# Figure
fig = plt.figure(figsize=(8,4))
figname = 'sigmab_35GHz.png'

ax1 = fig.add_subplot(1,1,1)
ax1.fill_between(req, sigmab_35_max, sigmab_35_min, color='0.8')
ax1.plot([], color='0.8', lw=6, label=r'Range for $1 \leq N_{\mathrm{cl}} = \xi^{3} \leq 125$')
ax1.plot(req, sigmab_35[0,:], '-', label=r'$N_{\mathrm{cl}} = 1$')
ax1.plot(req, sigmab_35[1,:], '-', label=r'$N_{\mathrm{cl}} = 4$')
ax1.plot(req, sigmab_35[2,:], '-', label=r'$N_{\mathrm{cl}} = 8$')
ax1.plot(req, sigmab_35[3,:], '-', label=r'$N_{\mathrm{cl}} = 16$')
ax1.plot(req, sigmab_35[4,:], '-', label=r'$N_{\mathrm{cl}} = 27$')
ax1.plot(req, sigmab_35[5,:], '-', label=r'$N_{\mathrm{cl}} = 64$')
ax1.plot(req, sigmab_35[6,:], '-', label=r'$N_{\mathrm{cl}} = 125$')
ax1.plot(req, sigmab_35[7,:], '-', color=[0.8,0.4,0], label=r'$N_{\mathrm{cl}} = 216$')
ax1.set_yscale("log")
ax1.grid()
ax1.set_xlim([0, 2.12])
xlabels = np.linspace(0,2,5)
ax1.set_xticks(xlabels)
ax1.set_xlabel(r'Mass-equivalent ice sphere radius $r_{\mathrm{eq}}$ [mm]', fontsize=12)
ax1.set_ylabel(r'$\sigma_{\mathrm{b}}$ [mm$^{2}$]', fontsize=12)
lgd = ax1.legend(loc='lower right', ncol=2, fontsize=11, borderpad=0.3, labelspacing=0.2, handlelength=2.0, handletextpad=0.3)
plt.tick_params(axis='both', labelsize=11)
plt.tight_layout()

#fig.savefig(figname)
plt.show()
