# -*- coding: utf-8 -*-
"""
Modeled snowfall triple-frequency radar signatures.

@author: mgergely
"""

import numpy as np
import matplotlib.pyplot as plt
import matplotlib.patches as patches


#######################################
#%% Data


# Data directories
inXKa = './X_Ka_data/'
inKuKa = './Ku_Ka_data/'
inKaW = './Ka_W_data/'

# Load DWRs [dB] for collections of randomly distributed ice spheres determined according to Sect. 3, at 10_35, 14_35, 35_94 GHz

#Select size distribution maximum diameter [mm] Dmax = 5.0 or 10.0 or 23.6
Dmax = 10.0

# Ncl = 1, 4, 8, 16, 27, 64, 125, 216 equivalent to xi = 1, 1.6, 2, 2.5, 3, 4, 5, 6
DWR_10_35 = np.loadtxt(inXKa + 'DWR_10_35_Ncl_Dlim_' + str(Dmax) + '.txt')
DWR_14_35 = np.loadtxt(inKuKa + 'DWR_14_35_Ncl_Dlim_' + str(Dmax) + '.txt')
DWR_35_94 = np.loadtxt(inKaW + 'DWR_35_94_Ncl_Dlim_' + str(Dmax) + '.txt')

# Total range = plume of many radar signatures for xi(D) following Eq. (7)
DWR_fullrange_10_35 = np.loadtxt(inXKa + 'DWR_10_35_fullrange_Dlim_' + str(Dmax) + '.txt')
DWR_fullrange_14_35 = np.loadtxt(inKuKa + 'DWR_14_35_fullrange_Dlim_' + str(Dmax) + '.txt')
DWR_fullrange_35_94 = np.loadtxt(inKaW + 'DWR_35_94_fullrange_Dlim_' + str(Dmax) + '.txt')


#######################################
#%% Plot X-, Ka-, W-band triple-frequency radar signatures for exponential size distribution of snowflake diameters D <= Dmax


# Create rectangles indicating large aggregates and rimed snowflakes, adapted from Kneifel et al. (2015)
rect1 = patches.Rectangle((2,4), 8, 10, linewidth=3, linestyle='solid', edgecolor=[0, 1, 1], facecolor='none', zorder=2)
rect2 = patches.Rectangle((3,-0.4), 11, 2.9, linewidth=3, linestyle='solid', edgecolor='m', facecolor='none', zorder=2)

# Figure
fig = plt.figure(figsize=(5,5))
figname = 'DWRs_X_Ka_W_Dlim_' + str(Dmax) + '.png'

ax1 = fig.add_subplot(1,1,1)
ax1.plot(DWR_fullrange_35_94.T, DWR_fullrange_10_35.T, '-', color='0.8', linewidth=5, zorder=1)
ax1.plot([], color='0.8', lw=6, label=r'$1 \leq N_{\mathrm{cl}}(D) = \xi^{3}(D) \leq 125$' + '\n  for Eq. (7)')
#ax1.plot(DWR_35_94.T, DWR_10_35.T, '-')
ax1.plot(DWR_35_94[0,:], DWR_10_35[0,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 1$')
ax1.plot(DWR_35_94[1,:], DWR_10_35[1,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 4$')
ax1.plot(DWR_35_94[2,:], DWR_10_35[2,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 8$')
ax1.plot(DWR_35_94[3,:], DWR_10_35[3,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 16$')
ax1.plot(DWR_35_94[4,:], DWR_10_35[4,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 27$')
ax1.plot(DWR_35_94[5,:], DWR_10_35[5,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 64$')
ax1.plot(DWR_35_94[6,:], DWR_10_35[6,:], '-', lw=2, label=r'$N_{\mathrm{cl}} = 125$')
ax1.plot(DWR_35_94[7,:], DWR_10_35[7,:], '-', lw=2, color=[0.8,0.4,0], label=r'$N_{\mathrm{cl}} = 216$')
ax1.add_patch(rect1)
ax1.add_patch(rect2)
ax1.grid()
ax1.set_xlim([-1,16.5])
ax1.set_ylim([-0.5,17])
ax1.text(8.5, 12.5, 'Aggregates', fontsize=11, bbox=dict(facecolor=[0, 1, 1], alpha=1.0))
ax1.text(10.35, 2.25, 'Graupel', fontsize=11, bbox=dict(facecolor='m', alpha=1.0))
ylabels = np.linspace(0,15,6)
ax1.set_xticks(ylabels)
ax1.set_yticks(ylabels)
ax1.set_xlabel('DWR Ka/W [dB]', fontsize=12)
ax1.set_ylabel('DWR X/Ka [dB]', fontsize=12)
lgd = ax1.legend(numpoints=2, loc='upper center', ncol=3, fontsize=11, borderpad=0.2, labelspacing=0.1, handlelength=1.5, handletextpad=0.2, borderaxespad=0.2, bbox_to_anchor=(0.50, 1.05))
plt.tick_params(axis='both', labelsize=11)
plt.tight_layout()

#fig.savefig(figname, bbox_extra_artists=(lgd,), bbox_inches='tight')
plt.show()
