# based on SW_Kviscosity.m

SWkviscosity = function(Temp,uTemp,Sal,uSal) {
  # SW_Kviscosity    Kinematic viscosity of seawater
  #=========================================================================
  # USAGE:  nu = SW_Kviscosity(Temp,uTemp,Sal,uSal)
  #
  # DESCRIPTION:
  #   Kinematic viscosity of seawater at atmospheric pressure (0.1 MPa) using
  #   dynamic viscosity and density correlations given in [1]
  #   Values at temperature higher than the normal boiling temperature
  #   are calculated at the saturation pressure.
  #
  # INPUT:
  #   Temp  = temperature
  #   uTemp = temperature unit
  #        'C'  : [degree Celsius] (ITS-90)
  #        'K'  : [Kelvin]
  #        'F'  : [degree Fahrenheit]
  #        'R'  : [Rankine]
  #   Sal  = salinity
  #   uSal = salinity unit
  #        'ppt': [g/kg]  (reference-composition salinity)
  #        'ppm': [mg/kg] (in parts per million)
  #        'w'  : [kg/kg] (mass fraction)
  #        '%'  : [kg/kg] (in parts per hundred)
  #
  #   Note: Temp and Sal must have the same dimensions
  #
  # OUTPUT:
  #   nu = specific kinematic viscosity [m^2/s]
  #
  #   Note: nu will have the same dimensions as Temp and Sal
  #
  # VALIDITY: 0 < Temp < 180 C and 0 < Sal < 150 g/kg;
  #
  # ACCURACY: 1.5% (estimated at average value within the range)
  #
  # REVISION HISTORY:
  #   2009-12-18: Mostafa H. Sharqawy (mhamed@mit.edu), MIT
  #               - Initial version
  #   2012-06-06: Karan H. Mistry (mistry@mit.edu), MIT
  #               - Allow Temp,Sal input in various units
  #               - Allow Temp,Sal to be matrices of any size
  #   2014-12-15: Daniel Neumann (daniel.neumann@hzg.de), Helmholtz-Zentrum Geesthacht
  #               - converted MATLAB code to R
  #
  # DISCLAIMER:
  #   This software is provided "as is" without warranty of any kind.
  #   See the file sw_copy.m for conditions of use and licence.
  #
  # REFERENCES:
  #   [1] M. H. Sharqawy, J. H. Lienhard V, and S. M. Zubair, Desalination
  #       and Water Treatment, 16, 354-380, 2010. (http://web.mit.edu/seawater/)
  #=========================================================================
  
  ## BEGIN ----
  mu  = SWviscosity(Temp,uTemp,Sal,uSal);
  rho = SWdensity(Temp,uTemp,Sal,uSal);
  nu  = mu / rho;
  
  return(nu)
}