! Numerical implementation of the contrail depth and ice crystal number parametrisation as described in Unterstrasser, ACP, 2016
! written by Simon Unterstrasser, DLR Oberpfaffenhofen
! If you want to use the program, please contact me: simon.unterstrasser@dlr.de
! The actual parametrisation is coded in the subroutine contrail_param.
! The main program contrail_properties contains various applications of the parametrisation and gives several examples of how to call the subroutine.
! The examples mainly reflect applications discussed in section 4 of Unterstrasser, ACP, 2016

program contrail_properties
  ! example calls of parametrisation routine "contrail_param"
  ! compilation at DLR with ifort contrail_properties.f90  
  implicit none

  ! ----------------------------------------------------------------------------
  ! Example I: 
  ! ----------------------------------------------------------------------------
  
  !  Temp and Rhi-values are chosen as in Block 1 simulation in Table a2 of the corresponding publication.

  integer,parameter::   nr=17
  real,Dimension(nr)::  Temp,RHi,EI,NBV,b,WVemit,Gamma0
  real::H,N,N_form
  integer::             i,j,k,l,m
  real::  Temp_2,RHi_2,NBV_2,b_2,EI_2
  real::N15=0.,N14=0.,N13=0.,N15form=0.,N14form=0.,N13form=0.
  real,DIMENSION(0:8,0:8,0:8,0:8)::N15vec,N14vec,N15formvec,N14formvec,H15vec,H14vec,Nconc15vec,Nconc14vec
  real,DIMENSION(0:8):: N15RHi, N15formRHi, N15Temp, N15formTemp, N15NBV, N15formNBV, N15span, N15formspan
  real,DIMENSION(0:8):: H15RHi, H15Temp, H15NBV, H15span, Nconc15RHi, Nconc15Temp, Nconc15NBV, Nconc15span
  real,DIMENSION(0:8):: N14RHi, N14formRHi, N14Temp, N14formTemp, N14NBV, N14formNBV, N14span, N14formspan
  real,DIMENSION(0:8):: H14RHi, H14Temp, H14NBV, H14span, Nconc14RHi, Nconc14Temp, Nconc14NBV, Nconc14span
  real,DIMENSION(0:8,0:8):: Nform_RHiEIiceno=0,N_RHiEIiceno=0,meanf_RHiEIiceno=0,Nsquare_RHiEIiceno=0
  real,DIMENSION(0:8,0:8):: Nform_TcaEIiceno=0,N_TcaEIiceno=0,meanf_TcaEIiceno=0,Nsquare_TcaEIiceno=0
  real,DIMENSION(0:8,0:8):: Nform_NBVEIiceno=0,N_NBVEIiceno=0,meanf_NBVEIiceno=0,Nsquare_NBVEIiceno=0
  real,DIMENSION(0:8,0:8):: Nform_spanEIiceno=0,N_spanEIiceno=0,meanf_spanEIiceno=0,Nsquare_spanEIiceno=0
  Temp = (/209.,209.,212.,212.,212.,217.,217.,217.,217.,217.,222.,222.,222.,222.,225.,225.,225./)
  RHi=   (/1.0 ,1.2 ,1.0 ,1.2 ,1.4 ,1.0 ,1.1,1.2 ,1.3,1.4,1.0 ,1.1,1.2 ,1.4,1.1,1.2 ,1.3/)
  EI= 2.8e14
  NBV=1.15e-2
  b=60.9
  WVemit=15e-3
  Gamma0=520
  do i=1,nr
    call contrail_param(Temp(i),RHi(i),EI(i),NBV(i),b(i),WVemit(i),Gamma0(i),H,N,N_form)
  enddo
  
!  OUTPUT of the program:
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  209  100   38.5  416  339  279    0
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  209  120   88.2  442  339  279  137
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  212  100   14.2  290  339  202    0
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  212  120   80.5  438  339  202  141
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  212  140   93.8  444  339  202  263
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  217  100    2.8   57  339  117    0
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  217  110   21.6  408  339  117   77
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  217  120   62.5  428  339  117  148
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  217  130   83.6  439  339  117  214
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  217  140   90.9  443  339  117  276
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  222  100    0.0    0  339   68    0
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  222  110   11.6  235  339   68   81
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  222  120   47.0  421  339   68  155
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  222  140   89.1  442  339   68  289
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  225  110    9.1  184  339   50   83
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  225  120   41.7  418  339   50  160
!    Temp, RHi, fNs, H, zdesc, zemit, zatm  225  130   76.9  436  339   50  231
 
! note that the listed values of the fNs and  H in Table (there denoted with a hat) can differ slightly from the values listed here.
! This is only due to round-off errors


  ! ----------------------------------------------------------------------------
  ! Example II: 
  ! ----------------------------------------------------------------------------
  !  described in Section 4.1 of the corresponding publication

!Check Reduction of EI_iceno by factor 10
!What is the effective reduction after the vortex phase
 ! RHi runs from 100% to 140% with increments of 5%
 ! Temp runs 210K to 226K with increments of 2K 
 ! NBV runs from 0.6.e-2s^-1 to 1.4e-2s^-1 with increments of 0.1e-2s^-1
 ! b runs from 20m to 84m with increments of 8m
 do i=0,8
   RHi_2=1.0+0.05*i
   do j=0,8
     Temp_2=210+j*2 
     do k=0,8
       NBV_2=0.1e-2*(6+k) 
       do l=0,8
         b_2=20+8*l 
         EI_2=1e15
         call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)
         N15 = N15 + N
         N15form = N15form + N_form
         
         EI_2=1e14
         call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)
         N14 = N14 + N
         N14form = N14form + N_form
         
         EI_2=1e13
         call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)
         N13 = N13 + N
         N13form = N13form + N_form
         
       enddo
     enddo
   enddo
 enddo
 N15=N15/(9*9*9*9)
 N14=N14/(9*9*9*9)
 N13=N13/(9*9*9*9)
 N15form=N15form/(9*9*9*9)
 N14form=N14form/(9*9*9*9)
 N13form=N13form/(9*9*9*9)
 print*,'EI15',N15form,N15,N15/N15form
 print*,'EI14',N14form,N14,N14/N14form
 print*,'EI13',N13form,N13,N13/N13form
 print*,'N15/N14,N15/N13,N14/N13',N15/N14,N15/N13,N14/N13
 
!  OUTPUT of the program:
!  EI15  7.8266447E+12  2.2797101E+12  0.2912755    
!  EI14  7.8265444E+11  4.3087348E+11  0.5505283    
!  EI13  7.8264861E+10  5.8328674E+10  0.7452728    
!  N15/N14,N15/N13,N14/N13   5.290904       39.08387       7.386992    

  ! ----------------------------------------------------------------------------
  ! Example III: 
  ! ----------------------------------------------------------------------------
  !  described in Section 4.1 of the corresponding publication

 do i=0,8
   RHi_2=1.0+0.05*i
   do j=0,8
     Temp_2=210+j*2 
     do k=0,8
       NBV_2=0.1e-2*(6+k) 
       do l=0,8
         b_2=20+8*l 
         do m=0,8 
           EI_2=1e12*10.0**(0.5*m)
           call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)

           Nform_RHiEIiceno(i,m) = Nform_RHiEIiceno(i,m) + N_form
           N_RHiEIiceno(i,m) = N_RHiEIiceno(i,m) + N
           Nsquare_RHiEIiceno(i,m) = Nsquare_RHiEIiceno(i,m) + N*N
           meanf_RHiEIiceno(i,m) = meanf_RHiEIiceno(i,m) + N/N_form

           Nform_TcaEIiceno(j,m) = Nform_TcaEIiceno(j,m) + N_form
           N_TcaEIiceno(j,m) = N_TcaEIiceno(j,m) + N
           Nsquare_TcaEIiceno(j,m) = Nsquare_TcaEIiceno(j,m) + N*N
           meanf_TcaEIiceno(j,m) = meanf_TcaEIiceno(j,m) + N/N_form

           Nform_NBVEIiceno(k,m) = Nform_NBVEIiceno(k,m) + N_form
           N_NBVEIiceno(k,m) = N_NBVEIiceno(k,m) + N
           Nsquare_NBVEIiceno(k,m) = Nsquare_NBVEIiceno(k,m) + N*N
           meanf_NBVEIiceno(k,m) = meanf_NBVEIiceno(k,m) + N/N_form

           Nform_spanEIiceno(l,m) = Nform_spanEIiceno(l,m) + N_form
           N_spanEIiceno(l,m) = N_spanEIiceno(l,m) + N
           Nsquare_spanEIiceno(l,m) = Nsquare_spanEIiceno(l,m) + N*N
           meanf_spanEIiceno(l,m) = meanf_spanEIiceno(l,m) + N/N_form
         enddo
       enddo
     enddo
   enddo
 enddo

 Nform_RHiEIiceno  = Nform_RHiEIiceno/(9*9*9)
 N_RHiEIiceno      = N_RHiEIiceno/(9*9*9)
 Nsquare_RHiEIiceno= Nsquare_RHiEIiceno/(9*9*9)
 meanf_RHiEIiceno  = meanf_RHiEIiceno/(9*9*9)

 Nform_TcaEIiceno  = Nform_TcaEIiceno/(9*9*9)
 N_TcaEIiceno      = N_TcaEIiceno/(9*9*9)
 Nsquare_TcaEIiceno= Nsquare_TcaEIiceno/(9*9*9)
 meanf_TcaEIiceno  = meanf_TcaEIiceno/(9*9*9)

 Nform_NBVEIiceno  = Nform_NBVEIiceno/(9*9*9)
 N_NBVEIiceno      = N_NBVEIiceno/(9*9*9)
 Nsquare_NBVEIiceno= Nsquare_NBVEIiceno/(9*9*9)
 meanf_NBVEIiceno  = meanf_NBVEIiceno/(9*9*9)

 Nform_spanEIiceno  = Nform_spanEIiceno/(9*9*9)
 N_spanEIiceno      = N_spanEIiceno/(9*9*9)
 Nsquare_spanEIiceno= Nsquare_spanEIiceno/(9*9*9)
 meanf_spanEIiceno  = meanf_spanEIiceno/(9*9*9)

 open (13,FILE='Datenmatrix_N_RHi_EIiceno.txt')
 write (13,*)  N_RHiEIiceno
 write (13,*)  Nform_RHiEIiceno
 write (13,*)  Nsquare_RHiEIiceno
 write (13,*)  meanf_RHiEIiceno
 close (13)

 open (13,FILE='Datenmatrix_N_Tca_EIiceno.txt')
 write (13,*)  N_TcaEIiceno
 write (13,*)  Nform_TcaEIiceno
 write (13,*)  Nsquare_TcaEIiceno
 write (13,*)  meanf_TcaEIiceno
 close (13)

 open (13,FILE='Datenmatrix_N_NBV_EIiceno.txt')
 write (13,*)  N_NBVEIiceno
 write (13,*)  Nform_NBVEIiceno
 write (13,*)  Nsquare_NBVEIiceno
 write (13,*)  meanf_NBVEIiceno
 close (13)

 open (13,FILE='Datenmatrix_N_span_EIiceno.txt')
 write (13,*)  N_spanEIiceno
 write (13,*)  Nform_spanEIiceno
 write (13,*)  Nsquare_spanEIiceno
 write (13,*)  meanf_spanEIiceno
 close (13)
  ! ----------------------------------------------------------------------------
  ! Example IV: 
  ! ----------------------------------------------------------------------------
  !  described in Section 4.2 of the corresponding publication

!Check Sensitivity to the input parameters
!What is the effective reduction after the vortex phase
 ! RHi runs from 100% to 140% with increments of 5%
 ! Temp runs 210K to 226K with increments of 2K 
 ! NBV runs from 0.6.e-2s^-1 to 1.4e-2s^-1 with increments of 0.1e-2s^-1
 ! b runs from 20m to 84m with increments of 8m
 do i=0,8
   RHi_2=1.0+0.05*i
   do j=0,8
     Temp_2=210+j*2 
     do k=0,8
       NBV_2=0.1e-2*(6+k) 
       do l=0,8
         b_2=20+8*l 
         EI_2=1e15
         call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)
         N15vec(i,j,k,l)= N
         N15formvec(i,j,k,l)= N_form
         H15vec(i,j,k,l)= H
          if (N.gt.0.0.AND.H.gt.0) Nconc15vec(i,j,k,l)= N/(H *0.63*b_2)*1e-6
!         if (N.gt.0.0.AND.H.gt.0) Nconc15vec(i,j,k,l)= N/(H *36)*1e-6

         EI_2=1e14
         call contrail_param(Temp_2,RHi_2,EI_2,NBV_2,b_2,0.0,0.0,H,N,N_form)
         N14vec(i,j,k,l)= N
         N14formvec(i,j,k,l)= N_form
         H14vec(i,j,k,l)= H
          if (N.gt.0.0.AND.H.gt.0) Nconc14vec(i,j,k,l)= N/(H *0.63*b_2) *1e-6
!         if (N.gt.0.0.AND.H.gt.0) Nconc14vec(i,j,k,l)= N/(H *36) *1e-6
        
       enddo
     enddo
   enddo
 enddo
write (6,*) '-------------------------------------------'
write (6,*) Nconc15vec
write (6,*) 'A-------------------------------------------A'
write (6,*) Nconc14vec
write (6,*) 'B-------------------------------------------B'
 N15RHi = SUM(SUM(SUM(N15vec,4),3),2)/(9*9*9)
 N15formRHi = SUM(SUM(SUM(N15formvec,4),3),2)/(9*9*9)
 H15RHi = SUM(SUM(SUM(H15vec,4),3),2)/(9*9*9)
 Nconc15RHi = SUM(SUM(SUM(Nconc15vec,4),3),2)/(9*9*9)
 write (6,*) Nconc15RHi

 N15Temp = SUM(SUM(SUM(N15vec,4),3),1)/(9*9*9)
 N15formTemp = SUM(SUM(SUM(N15formvec,4),3),1)/(9*9*9)
 H15Temp = SUM(SUM(SUM(H15vec,4),3),1)/(9*9*9)
 Nconc15Temp = SUM(SUM(SUM(Nconc15vec,4),3),1)/(9*9*9)
 write (6,*) Nconc15Temp

 N15NBV = SUM(SUM(SUM(N15vec,4),2),1)/(9*9*9)
 N15formNBV = SUM(SUM(SUM(N15formvec,4),2),1)/(9*9*9)
 H15NBV = SUM(SUM(SUM(H15vec,4),2),1)/(9*9*9)
 Nconc15NBV = SUM(SUM(SUM(Nconc15vec,4),2),1)/(9*9*9)
 write (6,*) Nconc15NBV

 N15span = SUM(SUM(SUM(N15vec,3),2),1)/(9*9*9)
 N15formspan = SUM(SUM(SUM(N15formvec,3),2),1)/(9*9*9)
 H15span = SUM(SUM(SUM(H15vec,3),2),1)/(9*9*9)
 Nconc15span = SUM(SUM(SUM(Nconc15vec,3),2),1)/(9*9*9)

 open (13,FILE='param_sens.txt')
 !open (13,FILE='param_sens_W36.txt')
  write (13,*) N15RHi
  write (13,*) N15formRHi
  write (13,*) N15Temp
  write (13,*) N15formTemp
  write (13,*) N15NBV
  write (13,*) N15formNBV
  write (13,*) N15span
  write (13,*) N15formspan

  write (13,*) H15RHi
  write (13,*) H15Temp
  write (13,*) H15NBV
  write (13,*) H15span

  write (13,*) Nconc15RHi
  write (13,*) Nconc15Temp
  write (13,*) Nconc15NBV
  write (13,*) Nconc15span

 N14RHi = SUM(SUM(SUM(N14vec,4),3),2)/(9*9*9)
 N14formRHi = SUM(SUM(SUM(N14formvec,4),3),2)/(9*9*9)
 H14RHi = SUM(SUM(SUM(H14vec,4),3),2)/(9*9*9)
 Nconc14RHi = SUM(SUM(SUM(Nconc14vec,4),3),2)/(9*9*9)
 write (6,*) Nconc14RHi

 N14Temp = SUM(SUM(SUM(N14vec,4),3),1)/(9*9*9)
 N14formTemp = SUM(SUM(SUM(N14formvec,4),3),1)/(9*9*9)
 H14Temp = SUM(SUM(SUM(H14vec,4),3),1)/(9*9*9)
 Nconc14Temp = SUM(SUM(SUM(Nconc14vec,4),3),1)/(9*9*9)
 write (6,*) Nconc14Temp

 N14NBV = SUM(SUM(SUM(N14vec,4),2),1)/(9*9*9)
 N14formNBV = SUM(SUM(SUM(N14formvec,4),2),1)/(9*9*9)
 H14NBV = SUM(SUM(SUM(H14vec,4),2),1)/(9*9*9)
 Nconc14NBV = SUM(SUM(SUM(Nconc14vec,4),2),1)/(9*9*9)
 write (6,*) Nconc14NBV

 N14span = SUM(SUM(SUM(N14vec,3),2),1)/(9*9*9)
 N14formspan = SUM(SUM(SUM(N14formvec,3),2),1)/(9*9*9)
 H14span = SUM(SUM(SUM(H14vec,3),2),1)/(9*9*9)
 Nconc14span = SUM(SUM(SUM(Nconc14vec,3),2),1)/(9*9*9)
 write (6,*) Nconc14span
  write (13,*) N14RHi
  write (13,*) N14formRHi
  write (13,*) N14Temp
  write (13,*) N14formTemp
  write (13,*) N14NBV
  write (13,*) N14formNBV
  write (13,*) N14span
  write (13,*) N14formspan

  write (13,*) H14RHi
  write (13,*) H14Temp
  write (13,*) H14NBV
  write (13,*) H14span

  write (13,*) Nconc14RHi
  write (13,*) Nconc14Temp
  write (13,*) Nconc14NBV
  write (13,*) Nconc14span

 close (13)

 open (13,FILE='Datenmatrix_N15_N14_Nconc15_Nconc14_H.txt')

 write (13,*)  N15vec
 write (13,*)  N14vec
 write (13,*)  Nconc15vec
 write (13,*)  Nconc14vec
 write (13,*)  H15vec

 close (13)


!  OUTPUT of the program:
!   7.81150945E+10   2.34461118E+11   5.49967888E+11   1.06441055E+12   1.79941605E+12   2.73318884E+12   3.76847965E+12   4.74706241E+12   5.54220650E+12
!    7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12
!    3.46367669E+12   2.93646290E+12   2.54243111E+12   2.25722656E+12   2.05584440E+12   1.91765571E+12   1.82710921E+12   1.77225558E+12   1.74464605E+12
!    7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12
!    1.44797113E+12   1.72439477E+12   1.96489445E+12   2.17359280E+12   2.35556490E+12   2.51552701E+12   2.65685801E+12   2.78275844E+12   2.89574643E+12
!    7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12   7.82666773E+12
!    4.93151060E+11   8.77823984E+11   1.32110929E+12   1.79769010E+12   2.28634290E+12   2.76977969E+12   3.23423778E+12   3.66940914E+12   4.06776421E+12
!    9.99999865E+11   1.95999983E+12   3.23999944E+12   4.84000032E+12   6.75999960E+12   9.00000016E+12   1.15599968E+13   1.44399999E+13   1.76399990E+13
!    55.3432312       138.558792       251.030045       351.306274       406.477142       426.109253       432.757874       436.687103       439.089966    
!    400.757599       374.118317       350.062531       329.971588       314.427216       302.753998       293.993652       287.824036       283.450928    
!    362.107239       353.284302       343.737213       334.317780       325.202087       316.454193       308.245544       300.605530       293.405914    
!    213.052719       260.952728       295.771271       322.244995       343.065765       359.360535       371.983704       381.785461       389.142548    
!    3.04502641E+10   9.00588749E+10   1.97577834E+11   3.44133206E+11   4.96924492E+11   6.09164722E+11   6.72956285E+11   7.07823272E+11   7.28769692E+11
!    7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11
!    5.70467287E+11   5.14243068E+11   4.68644168E+11   4.33277927E+11   4.06729687E+11   3.87407315E+11   3.73769404E+11   3.64552094E+11   3.58767591E+11
!    7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11
!    3.63099095E+11   3.87958473E+11   4.07878828E+11   4.24297759E+11   4.38110323E+11   4.49940947E+11   4.60201918E+11   4.69199946E+11   4.77171286E+11
!    7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11   7.82666564E+11
!    6.80401142E+10   1.28356483E+11   2.04331483E+11   2.94054691E+11   3.95796677E+11   5.07961967E+11   6.29061845E+11   7.57701280E+11   8.92554052E+11
!    1.00000006E+11   1.95999957E+11   3.23999957E+11   4.84000072E+11   6.76000039E+11   9.00000186E+11   1.15599986E+12   1.44400030E+12   1.76399948E+12
!    55.3432312       138.558792       251.030045       351.306274       406.477142       426.109253       432.757874       436.687103       439.089966    
!    400.757599       374.118317       350.062531       329.971588       314.427216       302.753998       293.993652       287.824036       283.450928    
!    362.107239       353.284302       343.737213       334.317780       325.202087       316.454193       308.245544       300.605530       293.405914    
!    213.052719       260.952728       295.771271       322.244995       343.065765       359.360535       371.983704       381.785461       389.142548    

END ! program contrail_properties
 
subroutine contrail_param(Temp,RHi,EI,NBV,b,WVemit,Gamma0,H,N,N_form)
!all inpupt and output variables are scalars
!output variables:
! H 		contrail height in m
! N 		number of ice crystals per meter (of flight path)
! input variables:
! Temp	temperature in K
! RHi	relative humidity w.r.t. ice, 1.0=100% 
! EI		number of ice crystal per kg fuel
! NBV	Brunt Visla frequency in s^-1
! b		wing span in m
! WVemit 	water vapor emission in kg/m, optional (if unspecified, set it to 0)
! Gamma0	initial circulation in m^2/s, optional (if unspecified, set it to 0)
! This program computes approximations of H and N of 5-minute old contrails for the given input parameters
! The approximations are based on the parametrisations presented in Unterstrasser, 2016, ACP, "Properties of young contrails"
! Note: If WVemit and/or Gamma0 are not available, simple estimates based on b are used. 
! assumptions
! cruise speed U =235m/s
!implicit none

real, INTENT(in):: Temp,RHi,EI,NBV,b,WVemit,Gamma0
real, INTENT(out):: N, H
real:: WVemitn,Gamma0n
real:: zdesc, zatm, zemit 
real:: pi = 3.14 
real:: rp, Ap ! plume radius rp (in m), plume area (in m^2)
real :: rho_emit ,N_form ! surplus WV concentration through emission
real, parameter:: beta0 = 0.45, beta1 = 1.19
real, parameter:: alpha0 = -1.35, alpha_desc=0.6, alpha_atm=1.7, alpha_emit=1.15
real::  alpha_atm_corr,alpha_emit_corr,EI_iceno_star_inv
real, parameter:: gamma_atm = 0.18, gamma_emit = 0.18
real, parameter:: EI_iceno_ref = 2.8e14  !(number of ice crystals per kg fuel)
real, parameter:: x_s=0.2, eta1=6.0, eta2=0.15     
real :: fNs, fNs_H, zDelta_N, zDelta_H   
!compute the three length scales zdesc, zatm and zemit
!   print*, 'Temp,RHi,EI,NBV,b,WVemit,Gamma0',Temp,RHi,EI,NBV,b,WVemit,Gamma0

  !compute zdesc
  Gamma0n=Gamma0
  if (Gamma0n.eq.0) Gamma0n = 10.*b-70.
  zdesc =  sqrt(8*Gamma0n/(pi*NBV))

  !compute zatm
  zatm = zbuffer(Temp,RHi,0.0)

  !compute zemit
     ! compute rho_emit = WV/Ap
       ! compute WV
       WVemitn=WVemit
       if (WVemitn.eq.0.) then 
          WVemitn = 3.1250e-06 *b*b 
       endif
       ! compute Ap
       rp= 1.5+0.314*b
       Ap=4*pi*rp*rp
     rho_emit = WVemitn/Ap
  zemit = zbuffer(Temp,1.0,rho_emit)
!  print*,'rp,Ap,rho_emit',rp,Ap,rho_emit

! compute zDelta  
  EI_iceno_star_inv  = EI_iceno_ref / EI
  alpha_atm_corr = alpha_atm * EI_iceno_star_inv**(gamma_atm)
  alpha_emit_corr = alpha_emit * EI_iceno_star_inv**(gamma_emit)
  
  zDelta_N = alpha_atm_corr * zatm + alpha_emit_corr * zemit - alpha_desc * zdesc
  zDelta_H = alpha_atm      * zatm + alpha_emit      * zemit - alpha_desc * zdesc  
 ! print*,' alpha_atm_corr, alpha_emit_corr , alpha_desc',alpha_atm_corr, alpha_emit_corr , alpha_desc
 ! print*,- alpha_desc * zdesc, alpha_atm_corr * zatm , alpha_emit_corr * zemit
 ! print*,'AA', zDelta_N,  alpha0+zDelta_N*0.01
     
! compute survival fraction fNs and total ice crystal number          
  
  fNs = beta0 + beta1/pi * atan(alpha0+zDelta_N*0.01)
  if (fNs.gt.1.) fNs = 1.0
  if (fNs.lt.0.) fNs = 0.0 
  N_form = WVemitn/1.25*EI
  N = N_form * fNs
     
! compute contrail depth

  fNs_H = beta0 + beta1/pi * atan(alpha0+zDelta_H*0.01)
  if (fNs_H.gt.1.) fNs_H = 1.0
  if (fNs_H.lt.0.) fNs_H = 0.0
  
  if (fNs_H.lt.x_s) then 
     H = zdesc * eta1*fNs_H
  else
     H = zdesc * (eta2*fNs_H + (eta1-eta2)*x_s)
  endif
  !  print '(A40,2I5,F7.1,4I5)' , 'Temp, RHi, fNs, H, zdesc, zemit, zatm',INT(Temp),INT(RHi*100), fNs*100, INT(H), INT(zdesc), INT(zemit), INT(zatm)
   !  print*, 'zdesc, zemit, zatm, fNs, H', zdesc, zemit, zatm, fNs*100, H
END ! routine contrail_param


real function zbuffer(Temp,RHi,rho_emit)
! input variables:
! temperature Temp
! relative humidity RHi
! emitted surplus water vapor concentration rho_emit
! output: zatm (if rho_emit is 0)
!         zemit (if RHi=1.0) 

! solve euqation:
! 1/R_v * (1+s_i) * e_s(T_CA)/T_CA +  rho_emit  = 1/R_v * e_s(T_CA + T_buffer) / (T_CA + T_buffer)
! z_buffer = T_buffer / Gamma_d
    implicit none
    real, INTENT(in)::Temp,RHi,rho_emit
    real :: R_v=  461.4  !J /(K * kg)
    real :: si,esat, rhov_links, rho_tot
    real :: dT(2)
    real::T,x
    integer::n_iterations,i
	si= RHi-1.0
	
	esat=exp(9.550426-5723.265/Temp+3.53068*alog(Temp)-0.00728332*Temp)  ! in Pa
	rhov_links= (1+si)*(esat)/(R_v*Temp) ! in kg/m^3
	rho_tot= rhov_links + rho_emit

	!bisection method
	dT(1)=0.0
	dT(2)=20.0
	
	n_iterations=19  ! -> epsilon = 10/(2^n_iterations) = 3.0518e-04 K in Temp
	
	do i=0,n_iterations  
		T=Temp+(dT(1)+dT(2))/2
		x= exp(9.550426-5723.265/(T)+3.53068*alog(T)-0.00728332*(T))/(R_v*T)   
		if (x.GE.rho_tot) then 
		  dT(2)=(dT(1)+dT(2))/2 
		else 
		  dT(1)=(dT(1)+dT(2))/2
		endif
	enddo
	if (dT(2).eq.20.0) then 
	   print*,'Choose a larger initial value for dT(2)!!'
	endif   
	!print*,(dT(1)+dT(2))/2,rho_emit
	zbuffer = ((dT(1)+dT(2))/2 / 9.8*1000.0  )
  END function ! function zbuffer
