! =======================================================================
! ===  Fortran code to calculate Onishi kernel                        ===
! ===                             Created by Ryo Onishi, January 2015 ===
! ===                                               updated July 2015 ===
! ===                                        onishi.ryo@jamstec.go.jp ===
! ===  Japan Agency for Marine-Earth Science and Technology (JAMSTEC) ===
! =======================================================================
!
!         Kcoag_Onishi=eta_e_Wang*Ec_Pinsky*sqrt(Kc_Onishi^2 + Kc_grav^2),
! where
!   eta_e_g   is the turbulence enhancement in Ec by Wang et al. (2008)
!   Ec_Pinsky is the collision efficiency by Pinsky et al. (2001)  
!   Kc_onishi is the Onishi kernel (without considering grav settling)
!   Kc_grav   is the gravitational (hydrodynamic) kernel.
!
! REFERENCES:
! For Onishi model
!   Onishi (2005) phD thesis, Kyoto Univ.
!   Onishi et al. (2006), Trans. JSME, 72, 2441-2448 (in Japanese)
!   Onishi et al. (2013), J. Comput. Phys., 242, 809-827
!   Onishi & Vassilicos (2014), J. Fluid Mech., 745, 279-299
!   Onishi et al. (2015), J. Atmos. Sci., 72, 2591-2607 (OMT2015)
! For Collision Efficiency
!   Pinsky et al. (2001), J. Atmos. Sci., 58, 742-764
! For Turbulence Enhancement of Collision Efficiency
!   Wang et al. (2008), New J. Phys., 10, 075013
!   Wang & Grabowski (2009) Atmos. Sci. Lett., 10, 1-8
!   
! =============================================================================

program main
  implicit none
  real :: radius1,radius2 ! radius [m]
  real :: Re_lam          ! Taylor-microscale-based Re
  real :: epsilon         ! energy dissipation rate [m2/s3]
  real :: nu              ! kinetic viscosity [m2/s]
  real :: ratio_rho       ! rho water/rho air
  real :: Kc_Onishi       ! collision kernel, Onishi
  real :: Ec_Pinsky       ! collision efficiency, Pinsky et al. (2001)
  real :: eta_e_Wang      ! turbulenct enhancement in Ec, eta_e, Wang et al. (2008)
  real :: Kcoag_Onishi    ! coagulation kernel, Onishi (E_coalescence=1)

!for Ayala comparison
  nu        = 1.7e-5 ![m2/s]
  ratio_rho = 1000.0
!for Onishi DNS comparison
!  nu        = 1.5e-5 ![m2/s]
!  ratio_rho = 843.0
  
  Re_lam  = 1000.
  epsilon = 0.04  ![m2/s3]
  
  radius1 = 30.e-6 ![m]
  radius2 = 30.e-6 ![m]

  print *, '! ====================================='
  print *, 'Re_lambda=',Re_lam
  print *, 'epsilon  =',epsilon
  print *, 'radius1  =',radius1
  print *, 'radius2  =',radius2
  
  !calculate Onishi kernel
  call calc_Kc_Onishi(radius1,radius2,Re_lam,epsilon,nu,ratio_rho,Kc_Onishi)

  !calculate Pinsky Ec
  call calc_Ec_Pinsky(radius1,radius2,Ec_Pinsky)

  !calculate Wang eta_e (2008 ver.): not to use!
  call calc_eta_e_New(radius1,radius2,epsilon,eta_e_Wang)

  !calculate Onishi coagulation kernel  
  Kcoag_Onishi = eta_e_Wang*Ec_Pinsky*Kc_onishi

  print *, 'Kc_Onishi   =',Kc_Onishi
  print *, 'Ec_Pinsky   =',Ec_Pinsky
  print *, 'eta_e_Wang  =',eta_e_Wang
  print *, 'Kcoag_Onishi=',Kcoag_Onishi

  stop
end program main


! #########################################################################################
! #########################################################################################
  subroutine calc_Kc_Onishi(r1,r2,Re_l,eps,dnu,ratio_den,Kc_total)
!   Kc=2PI*R^2*<|Wr|>g(R)
!
!   g11(R) is formulated in Onishi (2005) and Onishi et al. (2006).
!   Later the parameters were modified based on the new high-Re data obtained
!   by Onishi et al. (2013) and Onishi & Vassilicos (2014).
!   modified to better fit including St up to 8
!   modified to include gravity effect on Wr based on Onishi et al. (2009)
! Reference
!   Onishi (2005) phD thesis, Kyoto Univ.
!   Onishi et al. (2006), Trans. JSME, 72, 2441-2448 (in Japanese)
!   Onishi et al. (2009), Phys. Fluids, 21, 125108
!   Onishi et al. (2013), J. Comput. Phys., 242, 809-827
!   Onishi & Vassilicos (2014), J. Fluid Mech., 745, 279-299
!   Onishi et al. (2015), J. Atmos. Sci., 72, 2591-2607 (OMT2015)
!
!   For <|Wr|>: Wang et al. (2000), J. Fluid Mech., 415, 117-153
!   For g12(R): Zhou et al. (2001), J. Fluid Mech., 433, 77-104
! #########################################################################################
    implicit none
    real,intent(in) :: r1,r2     !radius [m]
    real,intent(in) :: Re_l      !Taylor-microscale-based Re
    real,intent(in) :: eps       !energy epsipation rate [m2/s3]
    real,intent(in) :: dnu       !kinetic viscosity [m2/s]
    real,intent(in) :: ratio_den !rho_droplet/rho_air
    real,intent(out) :: Kc_total
    
    real :: urms
    real :: leta
    real :: fallspeed1,fallspeed2
    real :: Kc_turb,Kc_grav
    real :: Wr, WrA2,WrS2
    real :: taup1,taup2,theta1,theta2,gamma
    real :: cw
    real :: CR,tauk,Te,phi
    real :: A1,A2
    real :: CA,alpha
    real :: StA,StB,St1,St2
    real :: y11,y21,y12,y22
    real :: za1,za2,xai
    real :: gR, gR1,gR2,RG12
    real,parameter :: PI=3.14159265 !3589793238462643383279502884
    real :: grav=9.80665 ![m/s2]
    real :: ftheta1,ftheta2,sig1,sig2
! -----------------------------------------------------------------
      urms  = sqrt(Re_l/sqrt(15./dnu/eps)) !RMS of u [m/s]
      CR    = r1+r2               !collision radius [m]
      taup1 = ratio_den * (2.*r1)**2 /18. /dnu !particle relaxation time [s]
      taup2 = ratio_den * (2.*r2)**2 /18. /dnu
      leta  = (dnu**3/eps)**0.25  !Kolmogorov scale [m]
      tauk  = leta**2/dnu         !Kolmogorov time  [s]
      Te    = Re_l*tauk/sqrt(15.)
      
      call termvel(r1,fallspeed1) !gravitational settling speed [m/s]
      call termvel(r2,fallspeed2)

! --- <|Wr|> in Wang et al. (2000)JFM
!updated to include gravity effect on Wr based on Onishi et al. (2009)PF @2015/07/07
      theta1 = 2.5* fallspeed1/grav /Te
      theta2 = 2.5* fallspeed2/grav /Te
      gamma = 0.183*urms**2/(dnu/leta)**2

      ftheta1 = gamma/(gamma-1.)                           &
               *( 1./(1.+theta1) - 1./(gamma*(1.+gamma*theta1)) )
      ftheta2 = gamma/(gamma-1.)                           &
               *( 1./(1.+theta2) - 1./(gamma*(1.+gamma*theta2)) )
      sig1 = fallspeed1/urms
      sig2 = fallspeed2/urms
      theta1=theta1 *sqrt((3.*(1.-ftheta1)+sig1**2)/(3.*(1.-ftheta1)+sig1**2))
      theta2=theta2 *sqrt((3.*(1.-ftheta2)+sig2**2)/(3.*(1.-ftheta2)+sig2**2))
!end of update @2015/07/07
!no grav      theta1 =2.5*taup1/Te
!no grav      theta2 =2.5*taup2/Te

      phi = max(theta2/theta1,theta1/theta2)
      cw = 1.+0.6*exp(-(phi-1.)**1.5)
      gamma = phi * gamma

      WrS2 = (dnu*CR)**2 /leta**4 /15.
      WrA2 = urms**2 * gamma/(gamma-1.)                           &
           *( (theta1+theta2)-4.*theta1*theta2/(theta1+theta2)     &
           *sqrt( (1.+theta1+theta2)/(1.+theta1)/(1.+theta2) ) ) &
           * ( 1./(1.+theta1)/(1.+theta2)                        &
              -1./(1.+gamma*theta1)/(1.+gamma*theta2) )
      WrA2 = cw*WrA2
      WrA2 = WrA2/3. !Ayala 2008
      Wr  =sqrt(2./PI*(WrA2 +WrS2))      
      
! --- g11(R) & g22(R), i.e., monodisperse
! --- empirical parameters for g11(R)
      A1 = 110.0
!      A2 = 0.38 !OMT2015
      A2 = 0.32 !updated
!      alpha = log10(0.26*Re_l**0.5)/log10(2.0) !OMT2015
      alpha = log10(0.23*Re_l**0.5)/log10(2.0) !updated
      alpha = max(alpha, 1.e-20)

!      CA = 0.06*Re_l**0.30 !OMT2015
      CA = 0.046*Re_l**0.36 !updated
      CA = min(CA, 0.094*Re_l**0.25) !added
      
      StA = (A2/A1)**0.25 * Re_l**0.25

      St1 = taup1/ tauk !Stokes number
      St2 = taup2/ tauk

      if(St2.le.StA) then
        y11 = A1 * St1**2
        y21 = 0.
      else
        y11 = 0.
        y21 = A2 * Re_l / St1**2
      endif
!
      if(St1.LE.StA) then
        y12 = A1 * St2**2
        y22 = 0.
      else
        y12 = 0.
        y22 = A2 * Re_l / St2**2
      endif
      
      za1 = 0.5 * (1. - tanh((log10(St1) - log10(StA)) / CA) )
      za2 = 0.5 * (1. - tanh((log10(St2) - log10(StA)) / CA) )

      gR1 = y11*za1**alpha + y21*(1.-za1)**alpha +1.
      gR2 = y12*za2**alpha + y22*(1.-za2)**alpha +1.

!g12=f(g11,g22) in Zhou et al. (2001)JFM      
      xai = max(taup2/taup1,taup1/taup2)     !xai:alpha
      RG12= 2.6*exp(-xai)+0.205*exp(-0.0206*xai) &
          * 0.5*(1.0+tanh(xai-3.0))
      gR = 1. + RG12*sqrt(gR1-1.)*sqrt(gR2-1.)

! --- Kc
      Kc_turb =2.*PI*CR**2 *Wr *gR     !turbulent Kc (no gravity)
      Kc_grav =PI*CR**2 *abs(fallspeed1-fallspeed2)  ! Kc,grav
! -----------------------------------------------------------------
      Kc_total=sqrt(Kc_turb**2 + Kc_grav**2)     ! Kc,total=(Kc,turb^2+Kc,grav^2)^0.5

!gR_onishi=gR
!wr_onishi=Kc_total/(2.*PI*CR**2 *gR)

    return
  end subroutine calc_Kc_Onishi
! #########################################################################################


! ###########################################################################
! ###########################################################################
  subroutine termvel(radius,fallvel)
! calculate the particle settling velocities
! called for setting the Hydrodynamic Kernel
! Table 1 in Beard (1976)JAS, 31, 851-864
! ###########################################################################
! terminal velocity of falling drops
      implicit none
      real,intent(out) :: fallvel ![m/s]
      real,intent(in)  :: radius ![m]
      real bb2(7),bb3(6)
      data bb2 /-0.318657e1,0.992696,-0.153193e-2,-0.987059e-3, &
               -0.578878e-3,0.855176e-4,-0.327815e-5/
      data bb3 /-0.500015e1,0.523778e1,-0.204914e1, &
               0.475294,-0.542819e-1,0.238449e-2/
      real eta0,el0,rhow,rhoa,grav,t0,sigma
      integer i
      real X,Y
      real C1,Csc,N_Da,N_Re,Np,Bo
      real d_cm ! diameter [cm]
      real vel_cmvs !fallvelocity in [cm/s]
! -----------------------------------------------------------------
      eta0 =1.818e-4 ![g/cm/s]
      el0 =6.62e-6   ![cm]
      rhow=1.0      ![g/cm^3]
      rhoa=1.225e-3 ![g/cm3]
      grav= 980.665 ![cm/s**2]
      t0  =273.15

! d_cm: diameter in cm-units
      d_cm=2.*radius*1.e2 ! [cm]
      if (d_cm.le.1.9e-3) then
        C1=grav*(rhow-rhoa)/(18.*eta0)
        Csc=1+2.51*el0/d_cm
        vel_cmvs=C1*Csc*d_cm**2
      elseif (d_cm.gt.1.9e-3.and.d_cm.le.1.07e-1) then
        N_Da=4.*rhoa*(rhow-rhoa)*grav/(3.*eta0**2)
        X=log(N_Da*d_cm**3)
        Y=0.
        do i=1,7
          Y=Y+bb2(i)*(X**(i-1))
        enddo
        Csc=1.+2.51*el0/d_cm
        N_Re=Csc*exp(Y)
        vel_cmvs=eta0*N_Re/(rhoa*d_cm)
      elseif (d_cm.gt.1.07e-1) then
        d_cm=min(d_cm,0.7) !limit
        sigma=76.1-0.155*(293.15-t0) !simple
! Eq.(5-12) in Pruppacher & Klett (1998)
! T_c=T-t0
! sigma=75.93+0.115*T_c+6.818e-2*T_c**2+6.511e-3*T_c**3+2.933e-4*T_c**4+6.283e-6*T_c**5+5.285e-8*T_c**6
        Bo=4.*(rhow-rhoa)*grav/(3.*sigma)*d_cm*d_cm
        Np=sigma**3*rhoa*rhoa/((eta0**4)*(rhow-rhoa)*grav)
        X=log(Bo*Np**(1./6.))
        Y=0.
        do i=1,6
           Y=Y+bb3(i)*(X**(i-1))
        enddo
        N_Re=Np**(1./6.)*exp(Y)
        vel_cmvs=eta0*N_Re/(rhoa*d_cm)
      endif
      
      fallvel=vel_cmvs*1.d-2 ! [cm/s]=>[m/s]
      
      return
      end subroutine termvel
! ###########################################################################


! ######################################################################
! ######################################################################
  subroutine calc_Ec_Pinsky(r1,r2,Ec_int)
! collision efficiency by Pinsky et al. (2001)JAS
! Table A1,A2&A3 at 1000hPa level
! ######################################################################
!
    real,intent(in) :: r1,r2
    real,intent(out) :: Ec_int !interpolated Ec
    real :: r_s, r_L !r_s < r_L
    integer :: k,ir,kk,iq
    real :: rratio,p,q
    real :: ecoll(13,12) !ecoll(ratio,r_L)
    real :: rat(13),r0(12)
    data r0 /5.e-6,10.e-6,15.e-6,20.e-6,25.e-6, &
             30.e-6,40.e-6,50.e-6,100.e-6,150.e-6, &
             200.e-6,250.e-6/ !collector radius[m]
    data rat /0.,0.05,0.1,0.2,0.3,0.4,0.5, &
              0.6,0.7,0.8,0.9,0.95,1.0/ !ratio r_s/r_L
! -----------------------------------------
    !ecoll(:,1) 5um
    ecoll(:,1)=(/0.0079,0.0079,0.0079,0.0079,0.01345,0.019,0.0225, &
                 0.026,0.0283,0.0306,0.0306,0.0306,0.0306/)
    !ecoll(:,2) 10um
    ecoll(:,2)=(/0.0029,0.0029,0.0029,0.0082,0.0131,0.0174,0.0206, &
                 0.0229,0.0247,0.026,0.0266,0.0266,0.0266/)
    !ecoll(:,3) 15um
    ecoll(:,3)=(/0.0013,0.0013,0.00265,0.0072,0.01245,0.019,0.02445, &
                 0.0279,0.0285,0.0272,0.0263,0.0266,0.0266/)
    !ecoll(:,4) 20um
    ecoll(:,4)=(/0.0008,0.0008,0.0025,0.0079,0.0229,0.062,0.1032, &
                 0.1202,0.107,0.069,0.0385,0.0333,0.0333/)
    !ecoll(:,5) 25um
    ecoll(:,5)=(/0.0006,0.000925,0.00285,0.0174,0.12305,0.2432,0.31425, &
                 0.3399,0.31975,0.2471,0.11635,0.068975,0.062/)
    !ecoll(:,6) 30um
    ecoll(:,6)=(/0.0006,0.0012,0.004,0.1007,0.3131,0.4475,0.5208, &
                 0.5465,0.5322,0.466,0.3088,0.17975,0.137/)
    !ecoll(:,7) 40um
    ecoll(:,7)=(/0.0008,0.0023,0.0378,0.4345,0.6428,0.7436,0.7915, &
                 0.8089,0.8054,0.7776,0.6874,0.5552,0.4192/)
    !ecoll(:,8) 50um
    ecoll(:,8)=(/0.0009,0.00745,0.2589,0.6713,0.816,0.8734,0.8992, &
                 0.9103,0.914,0.9029,1.0,1.0,1.0/)
    !ecoll(:,9) 100um
    ecoll(:,9)=(/0.0004,0.4475,0.8019,0.9404,0.9671,0.9786,1.0, &
                 1.0,1.0,1.0,1.0,1.0,1.0/)
    !ecoll(:,10) 150um
    ecoll(:,10)=(/0.0003,0.71715,0.9178,0.9748,1.0,1.0,1.0, &
                  1.0,1.0,1.0,1.0,1.0,1.0/)
    !ecoll(:,11) 200um
    ecoll(:,11)=(/0.0002,0.8553,0.9594,0.9864,1.0,1.0,1.0, &
                  1.0,1.0,1.0,1.0,1.0,1.0/)
    !ecoll(:,12) 250um
    ecoll(:,12)=(/0.0002,0.9234,0.9786,1.0,1.0,1.0,1.0, &
                  1.0,1.0,1.0,1.0,1.0,1.0/)
! -----------------------------------------
      
    r_s=min(r1,r2) !smaller radius [m]
    r_L=max(r1,r2) !Larger(collector) radius [m]

!radius class     
     do k=2,12 !11
       if (r_L.le.r0(k).and.r_L.ge.r0(k-1)) then
         ir=k
       elseif (r_L.gt.r0(11)) then
         ir=12 !11
       elseif (r_L.lt.r0(1)) then
         ir=1
       endif
     enddo

!ratio class
     rratio=r_s/r_L
     iq=2
     do kk=2,13
       if (rratio.le.rat(kk).and.rratio.gt.rat(kk-1)) iq=kk
     enddo

! two-dimensional linear interpolation of the collision efficiency
     if (ir.lt.11) then
       if (ir.ge.2) then
         p=(r_L-r0(ir-1))/(r0(ir)-r0(ir-1))
         q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
         Ec_int=(1.-p)*(1.-q)*ecoll(iq-1,ir-1)+ &
                     p*(1.-q)*ecoll(iq-1,ir)+        &
                     q*(1.-p)*ecoll(iq,ir-1)+        &
                     p*q*ecoll(iq,ir)
       else !ir==1
         q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
         Ec_int=(1.-q)*ecoll(iq-1,ir)+q*ecoll(iq,ir)
       endif
     else !ir==11
       q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
       Ec_int=(1.-q)*ecoll(iq-1,ir)+q*ecoll(iq,ir)
     endif

     return
  end subroutine calc_Ec_Pinsky
! ######################################################################


! #################################################################
! #################################################################
  subroutine calc_eta_e_New(r1,r2,eps,eta_e_int)
! eta_e by Wang et al. (2008)NJP
! updated using Table1 in Wang & Grabowski (2009)ASL
! modified by Ec(Pinsky)/Ec(Hall)
! #################################################################
!
    real,intent(in) :: r1,r2,eps
    real,intent(out) :: eta_e_int !interpolated eta_e
    real :: r_s, r_L !r_s < r_L
    real :: eta_e_E100,eta_e_E400, dd
    integer k,ir,kk,iq
    integer neps
    real rratio,p,q
    real eta_e(11,7,2) !eta_e(ratio,r_L,eps)
    real rat(11),r0(7)
    data r0 /10.e-6, 20.e-6, 30.e-6, 40.e-6, 50.e-6, 60.e-6, 100.e-6/ !WG2009
!    data rat /0.1,0.2,0.3,0.4,0.5, &
    data rat /0.0, 0.1,0.2,0.3,0.4,0.5, & !WG2009
              0.6,0.7,0.8,0.9,1.0/ !ratio r_s/r_L
! -----------------------------------------
    !ecoll(:,1,1) 10um (copy of 20um), eps=100cm2/s3 !WG2009
    eta_e(:,1,1)=(/1.740, 0.058, 2.673, 1.638, 0.995, 1.012, &
                          0.730, 0.710, 1.010, 1.631, 29.2/)
    !ecoll(:,1,2) 10um (copy of 20um), eps=400cm2/s3
    eta_e(:,1,2)=(/4.976, 0.119, 4.026, 1.952, 1.048, 0.795, &
                          0.832, 0.941, 1.418, 3.936, 22.6/)

    !ecoll(:,2,1) 20um, eps=100cm2/s3
    eta_e(:,2,1)=(/1.740, 0.058, 2.673, 1.638, 0.995, 1.012, &
                          0.730, 0.710, 1.010, 1.631, 29.2/)
    !ecoll(:,2,2) 20um, eps=400cm2/s3
    eta_e(:,2,2)=(/4.976, 0.119, 4.026, 1.952, 1.048, 0.795, &
                          0.832, 0.941, 1.418, 3.936, 22.6/)

    !ecoll(:,3,1) 30um, eps=100cm2/s3
    eta_e(:,3,1)=(/1.773, 0.711, 0.495, 0.623, 0.953, 1.056, &
                          1.112, 1.069, 1.175, 1.813, 6.41 /)
    !ecoll(:,3,2) 30um, eps=400cm2/s3
    eta_e(:,3,2)=(/3.593, 1.091, 0.586, 0.644, 0.973, 1.193, &
                          1.285, 1.286, 1.414, 2.187, 5.47/)

    !ecoll(:,4,1) 40um, eps=100cm2/s3 !WG2009
    eta_e(:,4,1)=(/1.490, 2.306, 1.292, 1.150, 1.112, 1.025, &
                          1.001, 0.983, 1.059, 1.340, 2.89/)
    !ecoll(:,4,2) 40um, eps=400cm2/s3
    eta_e(:,4,2)=(/2.519, 3.131, 1.511, 1.223, 1.143, 1.103, &
                          1.097, 1.099, 1.211, 1.507, 2.18/)

    !ecoll(:,5,1) 50um, eps=100cm2/s3
    eta_e(:,5,1)=(/1.207, 1.652, 0.745, 1.043, 1.064, 1.029, &
                          1.034, 0.991, 1.010, 1.305, 3.14/)
    !ecoll(:,5,2) 50um, eps=400cm2/s3
    eta_e(:,5,2)=(/1.445, 1.856, 0.857, 1.145, 1.100, 1.052, &
                          1.074, 1.023, 1.088, 1.343, 1.88/)

    !ecoll(:,6,1) 60um (copy of 50um), eps=100cm2/s3 !WG2009
    eta_e(:,6,1)=(/1.207, 1.652, 0.745, 1.043, 1.064, 1.029, &
                          1.034, 0.991, 1.010, 1.305, 3.14/)
    !ecoll(:,6,2) 60um (copy of 50um), eps=400cm2/s3
    eta_e(:,6,2)=(/1.445, 1.856, 0.857, 1.145, 1.100, 1.052, &
                          1.074, 1.023, 1.088, 1.343, 1.88/)

    !ecoll(:,7,1) 100um, eps=100cm2/s3 !WG2009
    eta_e(:,7,1)=(/1.0, 1.0,1.0,1.0,1.0,1.0, &
                        1.0,1.0,1.0,1.0,1.0/)
    !ecoll(:,7,2) 100um, eps=400cm2/s3
    eta_e(:,7,2)=(/1.0, 1.0,1.0,1.0,1.0,1.0, &
                   1.0,1.0,1.0,1.0,1.0/)
!! -----------------------------------------
      
    r_s=min(r1,r2) !smaller radius [m]
    r_L=max(r1,r2) !Larger(collector) radius [m]

! two-dimensional linear interpolation of the collision efficiency
    if (r_L.gt.r0(7)) then
      ir=8
    elseif (r_L.le.r0(1)) then
      ir=1
    else
      ir=2
      do k=2,7
        if (r_L.le.r0(k).and.r_L.gt.r0(k-1)) ir=k
      enddo
    endif

    rratio=r_s/r_L
    iq=2
    do kk=2,11
      if (rratio.le.rat(kk).and.rratio.gt.rat(kk-1)) iq=kk
    enddo

    if (ir.lt.8) then
      if(ir.ge.2) then
        p=(r_L-r0(ir-1))/(r0(ir)-r0(ir-1))
        q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
        eta_e_E100=(1.-p)*(1.-q)*eta_e(iq-1,ir-1,1)+ &
                       p*(1.-q)*eta_e(iq-1,ir,1)+        &
                       q*(1.-p)*eta_e(iq,ir-1,1)+        &
                       p*q*eta_e(iq,ir,1)
        eta_e_E400=(1.-p)*(1.-q)*eta_e(iq-1,ir-1,2)+ &
                       p*(1.-q)*eta_e(iq-1,ir,2)+        &
                       q*(1.-p)*eta_e(iq,ir-1,2)+        &
                       p*q*eta_e(iq,ir,2)
      else !ir==1
        q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
        eta_e_E100=(1.-q)*eta_e(iq-1,1,1) +q*eta_e(iq,1,1)
        eta_e_E400=(1.-q)*eta_e(iq-1,1,2) +q*eta_e(iq,1,2)
      endif
    else !ir==8
      q=(rratio-rat(iq-1))/(rat(iq)-rat(iq-1))
      eta_e_E100=(1.-q)*eta_e(iq-1,7,1) +q*eta_e(iq,7,1)
      eta_e_E400=(1.-q)*eta_e(iq-1,7,2) +q*eta_e(iq,7,2)
    endif

!    dd = min(eps,400.d-4)
    dd = min(eps,600.d-4) !WG2009 treatment
    if (dd.lt.100.d-4) then
      eta_e_int = 1. + (eta_e_E100 - 1.) * dd/100.e-4
    else         
      eta_e_int = eta_e_E100 * (dd-400.d-4)/(100.d-4-400.d-4) &
            + eta_e_E400 * (dd-100.d-4)/(400.d-4-100.d-4)
    end if
      
    return
  end subroutine calc_eta_e_New
! #################################################################

