#! /bin/tcsh -ef
################################################################################
# set computer specific shell variables
# set gnuplot path
set gnuplot="/usr/local/bin/gnuplot"
set gnuplot="/opt/local/bin/gnuplot"
# set ps2pdf path
set ps2pdf="/opt/local/bin/ps2pdf"
set ps2pdf="/usr/local/bin/ps2pdf"
# set open command to view pdf file
set open="ls -l"
set open="/usr/bin/open"
# set compiler with options
set f90="ifort -cpp -g -O0 -traceback -check all -debug extended"
################################################################################
# choose compound "NH42SO4" or "NaCl"
set speci="NH42SO4"
set speci="NaCl"
################################################################################
set prog="test_vi_${speci}"
################################################################################
# compound specification ...
if (${speci} == "NaCl") then
  set ns="1.000e-6"        # solute "concentration" [mol] - w.r.t. per m3(air)
  set Ds="2170.000"        # density [kg/m3]
  set Ms="0.058440"        # molar mass [kg/mol]
  set Ws="0.264700"        # mass fraction solubility [-]
  set RHDrefTc="25.00"     # reference DRH T-coefficient [-]
  set RHDrefT0="0.7528"    # reference deliquescence reative humidity (DRH) at T0 [-]
endif
if (${speci} == "NH42SO4") then
  set ns="1.000e-6"        # solute "concentration" [mol] - w.r.t. per m3(air)
  set Ds="1770.000"        # density [kg/m3]
  set Ms="0.132140"        # molar mass [kg/mol]
  set Ws="0.433100"        # mass fraction solubility [-]
  set RHDrefTc="80.00"     # reference DRH T-coefficient [-]
  set RHDrefT0="0.7997"    # reference deliquescence reative humidity (DRH) at T0 [-]
endif
################################################################################
cat > ./${prog}.f90 <<EOF
PROGRAM test_vi
!
! *** automatically generated with test_vi.j ***
!
! AUTHOR
! ------
! Swen Metzger <swen.metzger@mpic.de>
!
!      Department of Atmospheric Chemistry,
!      Max Planck Institute for Chemistry, Mainz, Germany.
!      Copyright 2012+. All rights reserved.
!
!      References
!      ---------
!      [1] CRC Hanbook of Chemistry and Physics, 85th Edition, 2004-2005.
!
!      [2] S. Metzger, B. Steil, L. Xu, J. E. Penner, and J. Lelieveld: ACP, 2012;
!          New representation of water activity based on a single solute specific constant
!          to parameterize the hygroscopic growth of aerosols in atmospheric models.
!------------------------------------------------------------------------------------
IMPLICIT NONE
!
INTEGER,PARAMETER  :: dp = SELECTED_REAL_KIND(12,307)
!--------------------------------------------------------------------------------
INTEGER              :: i,j,k                       ! loop indices
INTEGER, PARAMETER   :: imax = 686                  ! maximum loop count (100 + 586 of Rose et. al)
INTEGER, PARAMETER   :: IUNIT = 1                   ! I/O Unit number - read
INTEGER, PARAMETER   :: OUNIT = 2                   ! I/O Unit number - write
!--------------------------------------------------------------------------------
REAL(dp),PARAMETER   :: RHDrefT0=${RHDrefT0}_dp     ! reference RHD [-]
REAL(dp),PARAMETER   :: RHDrefTc=${RHDrefTc}_dp     ! reference RHD T-coefficient [-]
!--------------------------------------------------------------------------------
REAL(dp),PARAMETER   :: Phi=3.141592653589793_dp    ! Phi [-]
REAL(dp),PARAMETER   :: T0=298.15_dp                ! reference temperature, T0 [K]
REAL(dp),PARAMETER   :: R=8.314_dp                  ! universal gas constant [J/K/mol]
REAL(dp),PARAMETER   :: sigma=0.0761_dp             ! surface tension of H2O at 273 K [N/m]
REAL(dp),PARAMETER   :: T=298.15_dp                 ! default 298.15 temperature, T [K]
!--------------------------------------------------------------------------------
! compound specification; thermodynamic data of [1]
REAL(dp),PARAMETER   :: ns=${ns}_dp                 ! solute "concentration" [mol] - w.r.t. per m3(air)
REAL(dp),PARAMETER   :: Ds=${Ds}_dp                 ! solute density [kg/m3]
REAL(dp),PARAMETER   :: Dw=997.1_dp                 ! water  density [kg/m3]
REAL(dp),PARAMETER   :: Ms=${Ms}_dp                 ! molar mass solute [kg/mol]
REAL(dp),PARAMETER   :: Mw=0.018015_dp              ! molar mass H2O  [kg/mol]
REAL(dp),PARAMETER   :: Ws=${Ws}_dp                 ! mass fraction solubility [-]; Ws=ms/(ms+mw); ms=mw/(1/Ws-1)
REAL(dp),PARAMETER   :: mwat=1._dp                  ! initial mass of H20 [1 kg]
REAL(dp),PARAMETER   :: msat=mwat/(1._dp/Ws-1._dp)  ! saturation mass [kg] that completely dissolves at equilbrium RHD
REAL(dp),PARAMETER   :: nsat=msat/Ms                ! number of moles of solute in a saturated solution at RHD
REAL(dp)             :: RHD,RHDref                  ! calculated and reference RHD at T and T0 [-]
REAL(dp)             :: sm                          ! solute molality at RHD [mol/kg(H2O)]
REAL(dp)             :: aw                          ! water activity [-]
REAL(dp)             :: A                           ! A term [-], Eq.(2) of [2]
REAL(dp)             :: B                           ! B term [-], Eq.(3) of [2]
REAL(dp)             :: vi                          ! calculated ion-ion interaction parameter [-], Eq.(1) of [2]
REAL(dp)             :: GF                          ! diameter growth factor [0-1], Eq.(A2) of [2]
REAL(dp)             :: Ke                          ! Kelvin term [-], Eq. (A1) of [2]
!------------------------------------------------------------------------------------
! calculate T-dependency of RHDref - Eq. 5c of [2]
RHDref = MAX(0._dp,MIN(1._dp,RHDrefT0*EXP(RHDrefTc*(1._dp/T-1._dp/T0))))
!------------------------------------------------------------------------------------
WRITE(*,'(A,F12.6A)')' determine vi from RHD at T=',T,' with bisection - Eq. 5b of [2]'
!------------------------------------------------------------------------------------
CALL rtbis_rhd(vi,RHDref,nsat,Ke,A,B)
!-----------------------------------------------------------------
! (re-)calculate RHD - Eq. (5b) of [2]
RHD = Ke/(A+Mw*vi*(nsat+B)**vi)
WRITE(*,'(A,3F12.6)')' ${speci} - RHDref,RHD,vi ',RHDref,RHD,vi
!
STOP
!_________________________________________________________________________________________________________________________________
CONTAINS
!_________________________________________________________________________________________________________________________________
SUBROUTINE rtbis_rhd(vi,RHD,sm,Ke,A,B)
   ! determine vi from RHD with bisection
   IMPLICIT NONE
   ! bisection parameter
   REAL(dp), PARAMETER    :: x1   =   0._dp, &   ! lower boundary of root intervall
                             x2   =  10._dp, &   ! upper boundary of root intervall
                             xacc =  1.E-06_dp   ! accuracy
   REAL(dp), INTENT(IN)   :: RHD,     &  ! relative humidity of deliquescence (0-1) [-]
                             sm          ! solute molality at RHD [mol/kg(H2O)]
   REAL(dp), INTENT(OUT)  :: vi,&        ! single solute specific constant [-]
                             Ke,&        ! Ke-term [-];  Eq.(A1) of [2]
                             A,&         ! A-term  [-];  Eq. (2) of [2]
                             B           ! B-term  [-];  Eq. (3) of [2]
   REAL(dp)               :: GF          ! GF-term [0-1];Eq.(A2) of [2]
   INTEGER, PARAMETER  :: jmax=40
   INTEGER             :: j
   REAL(dp)            :: fmid, f, dx, xmid
    fmid = rhd_func(x2,RHD,sm,Ke,A,B)
    f    = rhd_func(x1,RHD,sm,Ke,A,B)
    IF ( f*fmid >= 0._dp ) THEN
       WRITE(*,*) 'STOP: Root must be bracketed for bisection.'
       STOP
    END IF
    IF(f < 0._dp) THEN
       vi = x1
       dx = x2-x1
    ELSE
       vi = x2
       dx = x1-x2
    END IF
    DO j = 1,jmax
       dx   = dx  * 0.5_dp
       xmid = vi + dx
       fmid = rhd_func(xmid,RHD,sm,Ke,A,B)
       IF(fmid <= 0._dp ) vi = xmid
       IF((ABS(dx) < xacc) .OR. (fmid == 0._dp)) RETURN
    END DO
    WRITE(*,*)  'STOP: too many bisections'
    STOP
END SUBROUTINE rtbis_rhd
!_________________________________________________________________________________________________________________________________
REAL FUNCTION rhd_func(vi,RHD,sm,Ke,A,B)
   ! determine vi from RHD with bisection
   IMPLICIT NONE
   REAL(dp), INTENT(IN)   :: RHD,     &  ! relative humidity of deliquescence (0-1) [-]
                             sm          ! solute molality at RHD [mol/kg(H2O)]
   REAL(dp), INTENT(OUT)  :: Ke,&        ! Ke-term [-];  Eq.(A1) of [2]
                             A,&         ! A-term  [-];  Eq. (2) of [2]
                             B           ! B-term  [-];  Eq. (3) of [2]
   REAL(dp)               :: vi,&        ! single solute specific constant [-]
                             GF,&        ! GF-term [0-1];Eq.(A2) of [2]
                             rhd_det     ! calculated RHD [-]
   ! growth factor - Eq. (A2) of [2]
   GF = (Ds/(Ms*Dw*sm)+1._dp)**(1._dp/3._dp)
   ! no Kelvin term required
   Ke = 1._dp
   ! calculate A-term from sm using Eq. (2) of [2]
   A = exp(-Mw*vi*sm**vi)*(1._dp+Mw*vi*sm)
   ! B-term - Eq. (3) of [2]
   B = 1._dp/(1._dp+1._dp/(Ms*vi*sm))*(vi*sm)**(-1._dp/vi)
   ! RHD - Eq. (5b) of [2]
   rhd_det  = Ke/(A+Mw*vi*(sm+B)**vi)
   rhd_func = rhd_det - RHD
END FUNCTION rhd_func
!_________________________________________________________________________________________________________________________________
END PROGRAM test_vi
EOF
################################################################################
# remove executable
rm -f ./a.out
# compile program
${f90} ./${prog}.f90
# excecute program
./a.out
################################################################################
# clean up
rm -f ./a.out #./${prog}.f90
################################################################################
exit

